breed [top-electrons top-electron]
breed [bottom-electrons bottom-electron]
breed [nuclei nucleus ]

globals [
  top-current bottom-current  ;; current measurements, accumulate for 100 ticks
  last-top-current last-bottom-current  ;; from previous 100 tick period
]

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  clear-all
  set-default-shape top-electrons "circle 2"
  set-default-shape bottom-electrons "circle 2"
  
  set-default-shape nuclei "circle 2"
  ;; create the top wire
  ask patches with [top-wire?]
    [ set pcolor gray ]
  ;; create the bottom wire
  ask patches with [not top-wire?]
    [ set pcolor gray + 3 ]

  ;; set up battery negative
  ask patches with [pxcor >= max-pxcor - 4]
    [ set pcolor red ]
  ask patch (max-pxcor - 1) (max-pycor / 2)
    [ set plabel "-" ]

  ;; set up battery positive
  ask patches with [pxcor <= min-pxcor + 4]
    [ set pcolor black ]
  ask patch (min-pxcor + 4) (max-pycor / 2)
    [ set plabel "+" ]

  ;; create electrons in top wire
  ask n-of (100000 / resistance-top-wire) patches with [top-wire?]
    [ sprout-top-electrons 1 [
        set color orange - 2
        set size 1
        if pcolor = black [
        set xcor xcor - 6
      ] ] ]
  ;; create electrons in bottom wire
  ask n-of (100000 / resistance-bottom-wire) patches with [not top-wire?]
    [ sprout-bottom-electrons 1 [
        set color orange - 2
        set size 1
        if pcolor = black [
        set xcor xcor - 6
      ] ] ]
  ask n-of (resistance-top-wire) patches with [top-wire? and pycor > (max-pycor / 2 + 0.1) and pcolor != black and pcolor != red]
    [ sprout-nuclei 1 [
        set color blue
        set size 1.5
         ] ]
  ask n-of (resistance-bottom-wire) patches with [not top-wire? and pycor < (max-pycor / 2 - 0.1) and pcolor != black and pcolor != red]
    [ sprout-nuclei 1 [
        set color blue
        set size 1.5
         ] ]
end

to-report top-wire?  ;; turtle or patch procedure
  report pycor > max-pycor / 2
end

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Runtime Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  ask top-electrons
    [ move-electron 
      ;; contribute to current measurement
      if pcolor = black [ pu 
        set top-current top-current + 1
        hatch 1
        [ setxy (xcor - 8) ycor
        set color orange - 2
        ]
        die
      ] ]
  ask bottom-electrons
    [ move-electron 
      ;; contribute to current measurement
      if pcolor = black [ pu 
        set bottom-current bottom-current + 1
        hatch 1
        [ setxy (xcor - 8) ycor
        set color orange - 2
        ]
        die
        
      ] ]
  tick
  ;if ticks mod 30 = 0 [ 
  update-plots 
  ;]
end

;; perform simple point collisions
;; with nuclei in the wire and steadily drifting
;; forward due to the electric field
to move-electron 
let old-patch patch-here
let old-xcor xcor 
let old-ycor ycor 
ifelse not any? nuclei-on neighbors 
 [
   set heading 270 fd 0.5 * voltage 
   ;set heading random 360 fd 0.3 
 ]
 [
   ifelse random-float 1.0 > 0.5 
   [ set heading 2 * heading ;; change direction due to collision 
     fd 0.5 * Voltage       ;; after the collsion, move by the same amount prior to the collision
     set heading random 360 ;; these two rules are just so that electrons don't get trapped between nuclei 
     fd 0.01    
   ]
   [ set heading 180 - 2 * heading ;; change direction due to collision 
     fd 0.5 * Voltage       ;; after the collsion, move by the same amount prior to the collision
     set heading random 360 ;; these two rules are just so that electrons don't get trapped between nuclei 
     fd 0.01    
   ]
 ] 
  ;; have we entered the wrong wire? if so, wrap
  if top-wire? != [top-wire?] of old-patch
    [ setxy old-xcor old-ycor ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plotting Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;;;

to update-plots
  set-current-plot "Current in top wire"
  plotxy (ticks) (top-current / ticks)
;  set last-top-current top-current / 30
;  set top-current 0
;  let top-max plot-y-max
  set-current-plot "Current in bottom wire"
  plotxy (ticks) (bottom-current / ticks)
;  set last-bottom-current bottom-current / 30
;  set bottom-current 0
;  let bottom-max plot-y-max
;  ;; make sure both plots are scaled the same amount
;  set-current-plot "Current in top wire"
;  set-plot-y-range 0 max list top-max bottom-max
;  set-current-plot "Current in bottom wire"
;  set-plot-y-range 0 max list top-max bottom-max
  end
@#$#@#$#@
GRAPHICS-WINDOW
236
10
930
245
85
-1
4.0
1
30
1
1
1
0
1
0
1
-85
85
0
50
1
1
1
ticks

CC-WINDOW
5
422
1065
517
Command Center
0

BUTTON
28
10
112
50
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
114
10
199
50
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
4
173
224
206
voltage
voltage
0.5
2
2
0.1
1
NIL
HORIZONTAL

PLOT
568
249
786
408
Current in bottom wire
time
current
0.0
0.0
0.0
1.0
true
false
PENS
"default" 1.0 0 -16777216 true

TEXTBOX
238
248
350
266
Battery-positive
11
0.0
0

TEXTBOX
843
248
935
270
Battery-negative\\n
11
0.0
0

SLIDER
3
138
225
171
resistance-bottom-wire
resistance-bottom-wire
100
500
250
1
1
NIL
HORIZONTAL

SLIDER
3
103
225
136
resistance-top-wire
resistance-top-wire
100
500
500
1
1
NIL
HORIZONTAL

PLOT
337
249
555
408
Current in top wire
time
current
0.0
0.0
0.0
1.0
true
false
PENS
"default" 1.0 0 -16777216 true

MONITOR
937
58
1055
103
no. of electrons
count top-electrons
3
1
11

MONITOR
939
187
1055
232
no. of electrons
count bottom-electrons
3
1
11

BUTTON
937
103
1055
136
watch an electron
ask one-of top-electrons with [xcor > max-pxcor - 10]\\n[ set color yellow \\n  pd\\nwatch-me\\n]  \\n \\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
939
233
1055
266
watch an electron
ask one-of bottom-electrons with [xcor > max-pxcor - 10]\\n[ set color green - 2 \\n  pd\\n  watch-me\\n] \\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

TEXTBOX
32
62
207
90
NOTE: \\"Resistance\\" is the same as \\"collision-rate-with-nuclei\\"
11
0.0
0

MONITOR
954
348
1044
393
total current
last-top-current + last-bottom-current
3
1
11

MONITOR
937
13
1055
58
Current In Top Wire
top-current / ticks
3
1
11

MONITOR
938
141
1056
186
Current in Bottom Wire
bottom-current / ticks
3
1
11

@#$#@#$#@
VERSION
-------
$Id: Parallel Circuit With Collisions.nlogo 39507 2008-04-24 15:23:31Z everreau $


WHAT IS IT?
-----------
This model offers a microscopic view of free electrons in two conducting wires that are connected in parallel to each other across two terminals of a battery.

This model shows what happens when two such wires are connected in parallel to each other across two ends of a battery terminal. It shows that current in each wire is not always equal to current in the other wire, unlike in a series circuit (see Series Circuit model). However, since each of the wires is connected across the same battery terminals, voltage is the same in each wire.


HOW IT WORKS
------------
The rules in this model are the same as in the earlier Ohm's Law and Series Circuit models.  As in Series Circuit, there are two wires, each with its own resistance, but here the wires are connected side by side rather than end to end.  Electrons from one wire do not cross over into the other wire.


HOW TO USE IT
-------------
The TOTAL-ELECTRONS slider allows you to select the total number of free electrons in both wires. This number is constant for a single run of the model.

The VOLTAGE slider controls the potential difference across the two battery terminals. This imparts a steady velocity to the electrons. However, this velocity is also dependent on the rate at which the electrons collide with the atomic nuclei in the wires.

The COLLISION-RATE-WITH-NUCLEI sliders, one for each wire, are inversely proportional to how far an electron travels on average without colliding with atomic nuclei. The collision rate of electrons in a wire causes resistance. The collision-rate affects the motion of electrons in it in another way: the net velocity of the electrons is also reduced in proportion to the collision rate.


THINGS TO NOTICE
----------------
When you observe the trace of the path of an electron, how does it differ in the two wires? Why?

What happens to the number of electrons in each wire when you change the collision rate of electrons in either of the wires?


THINGS TO TRY
-------------
1. Run the model with the default settings. Note the current in both the wires. Are these values equal? What about the number of electrons in each wire?

2. Increase the collision rate in one of the wires. Note the current in both the wires. Then increase the collision rate in both the wires at the same time. Note the current in both the wires. Is current in each wire still equal to each other? What about the number of electrons in each wire?

3. Watch a single electron in the top wire by pressing the top WATCH AN ELECTRON button. Now watch the tick counter and note how much model time the electron takes to travel through the wire. Repeat this observation several times for different values of the collision rates in each wire.
a) What do you notice?
b) Given the total number of electrons in each wire and the length of the wires, how can you calculate current in each wire by noting the time an electron takes to travel through the wire?

4. Following step 3 above, now calculate the current in the bottom wire.

5. Look in the section titled "Procedures for Counting Current" in the Procedures tab. How is current in each wire calculated in this model? Is this method and 3(b) equivalent to each other?

6. How would you calculate the total current in the circuit?


EXTENDING THE MODEL
-------------------
Can you create another wire in series with these two wires?


NETLOGO FEATURES
----------------
Electrons automatically wrap around the world horizontally. Special vertical wrap code is used to keep electrons from changing wires.


RELATED MODELS
--------------
Electrostatics
Ohm's Law
Series Circuit


CREDITS AND REFERENCES
----------------------
This model is a part of the NIELS curriculum. The NIELS curriculum is currently under development at Northwestern's Center for Connected Learning and Computer-Based Modeling. For more information about the NIELS curriculum please refer to http://ccl.northwestern.edu/NIELS.

Thanks to Daniel Kornhauser for his work on the design of this model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300
Circle -7500403 false true -45 -45 180
Circle -16777216 false false -2 -2 304

circle 2
false
0
Circle -16777216 true false 0 0 300
Circle -7500403 true true 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1pre1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="50"/>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="total-electrons">
      <value value="470"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="drift-velocity">
      <value value="0.4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-walk">
      <value value="0.4"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
