breed [ workers worker ]
breed [ firms firm ]

globals [
  tiers ;; stores the who of workers with certain levels of education
  req-education ;; stores the required education of workers for each tier
]

patches-own [
  tier-nbr
  firm-nbr
]

workers-own [
  offer ;; the greatest amount of money the worker has been offered
  pemployer ;; the potential employer of the worker
  ability
  salary
  education
  utility
  employer
  ecost
  expected-utility ;; expected utility of each tier
]

firms-own [
  sales ;; yearly sales the firm raises per tier
  cost ;; cost of operating per tier
  marginal-cost
  marginal-benefit
  wage-schedule ;; wage schedule
  revenue ;; revenue
]

to setup
  ca ;; clear all
  set tiers n-values types [ (list 1) ]
  set req-education n-values types [ 0 ]
  set-default-shape workers "person"
  setup-patches
  setup-workers ;; worker setup procedure
  setup-firms ;; firm setup procedure
  update-tiers
  setup-expected-utility
end

to setup-workers
  create-workers number-of-workers[
    setxy random-xcor random-ycor
    set ability ((random types) * 4 + 1)
    set education random-float types * 2
    set pemployer nobody
    set expected-utility n-values types [ 0 ]
    set color scale-color blue ability 0 (types * 4 + 1)
  ]
end

to setup-firms
  create-firms number-of-firms [
    set revenue n-values types [0]
    set cost n-values types [0]
    set sales n-values types [0]
    set marginal-cost n-values types [0]
    set marginal-benefit n-values types [0]
    set size 2
    set hidden? true
    setup-wage-schedule
  ]
end

to setup-expected-utility
  ask workers [
    foreach n-values types [?] [
      let expected-wage item ? [wage-schedule] of one-of firms
      let expected-educ item ? req-education
      set expected-utility replace-item ? expected-utility 
      ( expected-wage - expected-educ ^ education-difficulty / ability )
    ]
  ]
end

to setup-patches
  let yinterval (max-pycor - min-pycor) / number-of-firms
  let xinterval (max-pxcor - min-pxcor) / types
  foreach n-values number-of-firms [?] [
    let i ? + 1
    ask patches with [pycor - min-pycor <= i * yinterval and pycor - min-pycor >= (i - 1) * yinterval ] [
      set pcolor 3 + ? * 10
      set firm-nbr item ? n-values number-of-firms [? + number-of-workers] 
    ]]
  foreach n-values types [?] [
    let i (? + 1)
    ask patches with [pxcor - min-pxcor <= i * xinterval and pxcor - min-pxcor >= (i - 1) * xinterval ] [
      set tier-nbr ?
    ]
    ask patches [
      set pcolor scale-color pcolor tier-nbr -2 (number-of-firms + 3)
  ]]
end

;; we must assume that firms will give higher wages to people with more education
to setup-wage-schedule ;; firm procedure
  set wage-schedule n-values types [random-float types * initial-wage-difference]
  foreach n-values (types - 1) [?] 
  [ if ((item (? + 1) wage-schedule) - (item ? wage-schedule) < initial-wage-difference) [ setup-wage-schedule ] ]
end

to-report tier-of-worker [ n ] ;; reports the tier of the worker given his who number
  foreach tiers 
  [ if (member? [who] of worker n ? = true) [ report position ? tiers ] ]
end

to-report workers-in-tier [ n ] ;; reports an agentset of the nth tier
  report workers with [ member? [who] of self item n tiers ]
end

;; From here the go procedure
;;  First, workers start with their initial values of education. Then, firms go and 
;;  make offers to workers. Next, workers accetp the highest offer. Now, workers and firms 
;;  calculate their utilities.

to go
  ask firms 
  [ update-tiers
    offer-wage ]
  ask workers 
  [ accept-offer
    update-utility
    update-expected-utility 
    ;move 
    ]
  ask firms 
  [ observe-performance 
    update-wage-schedule ]
  ask workers [ 
    update-education 
    move
  ]
  update-plots
  tick
end

to move 
  foreach n-values types [?] [
    ask workers-in-tier ? [
      if ([tier-nbr] of patch-here != tier-of-worker who) 
      [ move-to one-of patches with [tier-nbr = tier-of-worker [who] of myself] ]
      ifelse (employer != nobody)
      [ if ([firm-nbr] of patch-here != [who] of employer )
        [ move-to one-of patches with [tier-nbr = tier-of-worker [who] of myself and firm-nbr = [who] of [employer] of myself]] ]
      [ setxy min-pxcor min-pycor ] ;do nothing
      ]
    ]
end

to get-education
  set education education + 1
  set ecost education ^ education-difficulty / ability
end

to update-tiers ;; need to check if this works!!
  foreach n-values (types - 1) [?] [
    let a item ? req-education
    let b item (? + 1) req-education
    let this-tier [who] of workers with [education < b and education >= a]
    set tiers replace-item ? tiers this-tier
  ]
  let last-req last req-education
  let last-tier [who] of workers with [education >= last-req]
  set tiers replace-item (types - 1) tiers last-tier
  ;; now set the minimum level of education for each tier
  foreach n-values types  [?] [
;    if (any? workers-in-tier ?)
    set req-education replace-item ? req-education (0 + tier-length * ?) 
;    [ set mean-education-of-tiers replace-item ? req-education (mean [education] of workers-in-tier ?) ]
  ]
end

to offer-wage ;; firm procedure
  foreach n-values types [?] ;; for 1, 2, 3, ..., types. This is just a counting variable
  [
    let num ? ;; set num as a counting variable of the current number
    let w item num [wage-schedule] of self ;; wage-schedule = [w1 w2 w3]. num denotes current w.
    let T item num tiers ;; tiers = [T1 T2 T3]. num denotes current T
    foreach T [ ;; for T1, T2, ...
      ask worker ? [
        let temp w ;; offer wage of w
        if [offer] of self <= temp
        [ set offer temp 
          set pemployer myself ]]
  ]]
end

to accept-offer 
    set salary offer
    set employer pemployer
;    set employed? true
    set pemployer nobody
    set offer 0
end

to observe-performance ;; firm procedure
  foreach n-values types [?] [ ;; for each tier
    if (any? (workers-in-tier ?) with [employer = myself]) [
      let sum-salaries sum [salary] of (workers-in-tier ?) with [employer = myself]
      let sum-abilities sum [ability] of (workers-in-tier ?) with [employer = myself]
      let num-workers count (workers-in-tier ?) with [employer = myself]
      set cost replace-item ? cost sum-salaries
      set sales replace-item ? sales sum-abilities
      set marginal-cost replace-item ? marginal-cost (sum-salaries / num-workers)
      set marginal-benefit replace-item ? marginal-benefit (sum-abilities / num-workers)
    ]
  ]
  set revenue (map [?1 - ?2] sales cost)
end

to update-wage-schedule ;; firm procedure
  foreach n-values types [?] [
    let sum-ability sum [ability] of workers-in-tier ? ;; workers with their who number included in subtier
    let sum-salary sum [salary] of workers-in-tier ?
    ;; this is for the profit maximizing firm that tries to balance the wages with costs
    if (firm-type = "equate sales and cost") 
    [ 
      if (sum-salary >= sum-ability) [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule - step-size) ]
      if (sum-salary < sum-ability) [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule + step-size) ] 
    ]
    ;; this is for the marginal firm
    if (firm-type = "marginal analysis") 
    [
      ifelse (item ? marginal-cost > item ? marginal-benefit) 
        [set wage-schedule replace-item ? wage-schedule (item ? wage-schedule - step-size) ]
        [if (item ? marginal-cost < item ? marginal-benefit)
          [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule + step-size) ] 
        ]
    ]
    ;; this is for the rewarding firm
    if (firm-type = "rewarding company") 
    [ 
      if (sum-salary >= sum-ability) [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule + step-size) ]
      if (any? workers-in-tier ? = false) [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule + step-size) ] 
    ]
    ;; this is for the penalizing firm
    if (firm-type = "punishing company")
    [
      if (sum-salary < sum-ability) [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule - step-size) ]
      if (any? workers-in-tier ? = false) [ set wage-schedule replace-item ? wage-schedule (item ? wage-schedule + step-size) ]
    ]
  ]
end

to update-expected-utility
  foreach n-values types [?] [ ;; for n = [ 0 ... types ]
    ifelse (tier-of-worker who = ?) 
    ;; if the worker is in tier n
    [ set expected-utility replace-item ? expected-utility utility ];; set his expected utility 
    ;; if the worker is not in tier n
    [ if (any? workers-in-tier ?) [
      let mean-education item ? req-education 
      let observed-salary [salary] of one-of workers-in-tier ?
      let observed-utility (observed-salary - mean-education ^ education-difficulty / ability)
      set expected-utility replace-item ? expected-utility observed-utility ] ]
  ]
end

to update-utility;; worker procedure
  set ecost education ^ education-difficulty / ability
  set utility salary - ecost
end

;; workers optimize their education incrementally if they are in their highest earning bracket
;; but if not, they move to their highest earning bracket.
to update-education ;; worker procedure
  let my-tier tier-of-worker who
  let optimal-tier position (max expected-utility) expected-utility
  ifelse (optimal-tier = my-tier) 
  ;; if i am in the highest earning bracket
  [ ifelse (utility > 0) 
    [ if (education >= step-size and education > min [education] of workers-in-tier my-tier) [set education education - step-size ]]
    [ set education education + step-size ]  ]
  ;; if i am not in the highest earning bracket
  [ set education item optimal-tier req-education ]
end

to update-plots
  set-current-plot "Wage vs. Tier"
  plot-pen-reset
  foreach n-values types [?] [
    if (any? workers-in-tier ?) 
    [ plotxy ? mean [salary] of workers-in-tier ? ]
  ]
  
  set-current-plot "Education vs Tiers"
  plot-pen-reset
  foreach n-values types [?] [
    if (any? workers-in-tier ?) 
    [ plotxy ? mean [education] of workers-in-tier ?]
  ]
  
  set-current-plot "Ability vs Tiers"
  plot-pen-reset
  foreach n-values types [?] [
    if (any? workers-in-tier ?) 
    [ plotxy ? mean [ability] of workers-in-tier ?]
  ]
  
  set-current-plot "Mean Educations"
  plot-pen-reset
  foreach n-values types [?] [
    if (any? workers-in-tier ?) 
    [ plotxy ? mean [education] of workers-in-tier ?]
  ]
  
  set-current-plot "Utilities by Firm"
  plot-pen-reset
  foreach n-values number-of-firms [?] [
    plotxy ? sum [revenue] of firm (? + number-of-workers)
  ]  
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks

SLIDER
17
10
189
43
number-of-firms
number-of-firms
1
5
3
1
1
NIL
HORIZONTAL

SLIDER
16
48
192
81
number-of-workers
number-of-workers
1
100
49
1
1
NIL
HORIZONTAL

BUTTON
28
254
95
287
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
109
254
172
287
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
16
86
193
119
types
types
1
5
4
1
1
NIL
HORIZONTAL

SLIDER
16
162
192
195
step-size
step-size
0
0.05
0.01
0.001
1
NIL
HORIZONTAL

PLOT
656
13
856
163
Wage vs. Tier
tier
wages
0.0
1.0
0.0
1.0
true
false
PENS
"default" 1.0 1 -16777216 true

PLOT
657
167
857
317
Education vs Tiers
NIL
NIL
0.0
1.0
0.0
1.0
true
false
PENS
"default" 1.0 1 -16777216 true

SLIDER
16
124
192
157
initial-wage-difference
initial-wage-difference
0
20
10
1
1
NIL
HORIZONTAL

PLOT
657
323
857
473
Ability vs Tiers
NIL
NIL
0.0
1.0
0.0
1.0
true
false
PENS
"default" 1.0 1 -16777216 true

PLOT
861
12
1061
162
Mean Educations
NIL
NIL
0.0
1.0
0.0
1.0
true
false
PENS
"default" 1.0 1 -16777216 true

SLIDER
18
340
195
373
education-difficulty
education-difficulty
0
6
2
0.1
1
NIL
HORIZONTAL

SLIDER
18
298
194
331
tier-length
tier-length
0
15
3
0.1
1
NIL
HORIZONTAL

CHOOSER
16
198
196
243
firm-type
firm-type
"equate sales and cost" "marginal analysis" "rewarding company" "punishing company"
2

PLOT
862
167
1062
317
Utilities by Firm
Firms
Utility
0.0
1.0
0.0
1.0
true
false
PENS
"default" 1.0 1 -16777216 true

@#$#@#$#@
WHAT IS IT?
-----------
This model tests Michael Spence's model of job market signaling. In his model, he shows how firms may differentiate among two types of workers using education as a signal. Here, education has no direct benefit to either party: it does not make the worker's utility go up but rather decreases it, and it does not necessitate that firms can give the correct wages. However, both firms and workers soon find out that they can differentiate among workers and among themselves by getting different levels of education.


HOW IT WORKS
------------
Initially, workers will be divided into TIERS based on their level of education. At each tick, firms will offer workers WAGES. Workers will then choose the highest wage that was offered and work for that company. After the workers accept, firms will calculate their revenues by looking at the revenue each tier raises. Depending on the FIRM-TYPE, the firm chooses to increase or decrease the wages given to that tier at this point. Next, the workers have a choice to increase or decrease their education. They do this by calculating the tier that gives the maximum expected utility, and then setting their EDUCATION to the minimum amount of education needed to get into that tier, which is given by REQ-EDUCATION. When workers have updated their education levels, they move to the corresponding position in the view. 


HOW TO USE IT
-------------
NUMBER-OF-FIRMS, NUMBER-OF-WORKERS adjusts the number of workers and firms. TYPES is they number of different ABILITIES the worker has. INITIAL-WAGE-DIFFERENCE is the initial wage difference across tiers, which you can set to be anywhere from zero to twenty. TIER-LENGTH is the length of each tier in amounts of education, and EDUCATION-DIFFICULTY is how much more costly it is for workers to get a level of education. STEP-SIZE is the factor by which firms increase or decrease the wages of workers, and can be set to smaller values for more exact results. FIRM-TYPE denotes the type calculation the firm uses when adjusting wages. 


THINGS TO NOTICE
----------------
Each worker is colored differently, with the brightest workers denoting the workes with the most ability, and darker workers denoting workers with less ability. The patches are colored as well. Each colored row represents a different firm, the topmost being the firm with the highest number. Each colored column represents a different education tier, with the rightmost being the one with the highest education. Each worker moves to a patch that denotes its tier and its employer. Over time, the workers should separate themselves so that the darker workers move to darker patches, and vice versa. Sometimes, workers go to the bottom left corner. This means that they do not get hired, or that they do not accept their offer since it is too low. These workers are considered unemployed. 


THINGS TO TRY
-------------
Try varying the INITIAL-WAGE-DIFFERENCE, TIER-LENGTH, EDUCATION-DIFFICULTY to find conditions for which the workers can be correctly differentiated. Is this even possible? Under which conditions do the workers end up receiving the same wage? How is this different for each type of firm?


EXTENDING THE MODEL
-------------------
Currently, the model has it that the interval for each tier of education is constant. Try altering the model so that the firms can adjust the length of each interval independently. 


NETLOGO FEATURES
----------------
This model uses lists extensively. Since it was difficult to create and manipulate lists of agentsets, the who numbers of workers were used. Two functions were created to access the turtles of a given tier, and to access the tier of a given turtle. 


RELATED MODELS
--------------
To the extent of my knowledge, there are no related agent based models that have modeled Michael Spence's Job Market Signaling. There is, however, a plethora of adaptations of his theory in economics and business. 


CREDITS AND REFERENCES
----------------------
This model is inspired by a paper by Michael Spence. "Job Market Signaling", Michael Spence, The Quarterly Journal of Economics (1973) 87 (3): 355-374
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="500"/>
    <metric>[ability] of workers-in-tier 0</metric>
    <metric>[ability] of workers-in-tier 1</metric>
    <metric>[ability] of workers-in-tier 2</metric>
    <metric>[ability] of workers-in-tier 3</metric>
    <metric>[wage-schedule] of firms</metric>
    <enumeratedValueSet variable="step-size">
      <value value="0.01"/>
    </enumeratedValueSet>
    <steppedValueSet variable="tier-length" first="1" step="1" last="10"/>
    <enumeratedValueSet variable="number-of-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="firm-type">
      <value value="&quot;equate sales and cost&quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="types" first="2" step="1" last="5"/>
    <steppedValueSet variable="education-difficulty" first="1.5" step="0.5" last="3"/>
    <steppedValueSet variable="initial-wage-difference" first="1" step="1" last="10"/>
    <enumeratedValueSet variable="number-of-workers">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
