;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DECLARATIONS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
breed [ people ]


globals [
  time
  total-population

  ;internal influence counters and statistics
  contacts                ;; number of contacts made with other individuals
  old-contacts            ;; previous total
  contacts-per-time       ;; how many contacts made this time period
  conversation-starts     ;; total number of conversations
  old-conversation-starts ;; previous total
  starts-per-time         ;; how many conversations made this time period - number ended
  conversation-ends       ;; total number ended
  old-conversation-ends   ;; previous total
  ends-per-time           ;; how many conversations made this time period - number ended
  active-conversations    ;; total number of conversations currently active
  kind-structure          ;; used to determine type of adopter

  ;external influence counters and statistics
  externals               ;; total number of external mass media contacts
]

patches-own [
 original-color
]

people-own [
  status                 ;; "potential", "adopter" of technology, "disrupter" who adopted new technology
  kind                   ;; 1 - 5 are "innovator","early_adopter","early_majority","late_majority","laggard" assuming Rogers' percentages
  change-agent?          ;; change agent can meet with multiple people (TBD)
  initiator?             ;; keep track of who initiates the conversation
  partner                ;; The person that is our current partner in a conversation
  media?                 ;; Keep track of whether listening to mass media presentation
  start-time             ;; when current conversation starts
  end-time               ;; when current conversation will end
  num-conversations      ;; cumulative number of conversations
  total-interact-time    ;; cumulative time spent in conversation
  num-externals          ;; cumulative number of interactions with external media
  total-external-time    ;; cumulative time spent with external media
]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SETUP
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  ca
  setup-globals
  setup-env
  setup-people
  setup-plot
end

to setup-globals
  random-seed seed
  set time 0
  set total-population 0

  ;internal
  ;Note that kind-structure is based on Rogers' graphs, which is a normal distribution
  ;in which categories 1 & 2 = 5, while 3 = 4.  However category 1 is set by the number
  ;of adopters + disrupters - those who already have the new innovation.  So the table
  ;here is set to category 2 being the first category.  Adopters + Disrupters are
  ;set by sliders.
  set contacts 0
  set old-contacts 0
  set contacts-per-time 0.0
  set conversation-starts 0
  set old-conversation-starts 0
  set starts-per-time 0
  set conversation-ends 0
  set old-conversation-ends 0
  set ends-per-time 0
  set active-conversations 0
  set kind-structure [ [ 2 12 ] [ 3 47 ] [ 4 81 ] [ 5 100 ] ]

  ;external
  set externals 0
end

to setup-env
  ;patch is default initially
  ask patches [
    set pcolor black
    set original-color black
  ]

  ;adopter mass media center
  if e-adopt [
    let x int(x-adopt / 100 * max-pxcor)
    let y int(y-adopt / 100 * max-pycor)
    let s int(s-adopt / 100 * (max-pxcor + 1))
    ask patches [
      if (pxcor > x - s) and (pxcor < x + s)
         and (pycor > y - s) and (pycor < y + s) [
         set pcolor orange
         set original-color orange
      ]
    ]
  ]

  ;disrupter mass media center
  if e-disrupt [
    let x int(x-disrupt / 100 * max-pxcor)
    let y int(y-disrupt / 100 * max-pycor)
    let s int(s-disrupt / 100 * (max-pxcor + 1))
    ask patches [
      if (pxcor > x - s) and (pxcor < x + s)
         and (pycor > y - s) and (pycor < y + s) [
         set pcolor cyan
         set original-color cyan
      ]
    ]
  ]

end

to setup-people

  create-people initial-potentials [
      set total-population initial-potentials + total-population
      setup-people-parms
      setup-potential
      set kind determine-kind ( random 100 )
    ]

  if adopter-switch [
      set total-population initial-adopters + total-population
      create-people initial-adopters[
        setup-people-parms
        setup-adopter
        set kind 1
      ]
  ]

  if disrupter-switch [
    set total-population initial-disrupters + total-population
    create-people initial-disrupters [
       setup-people-parms
       setup-disrupter
       set kind 1
    ]
  ]
end

to setup-people-parms
    setxy random-pxcor random-pycor ; centers on patch in Version 3.1
    set end-time 0
    set total-interact-time 0
    set num-conversations 0
    set total-external-time 0
    set num-externals 0
    set partner nobody
    set shape "person"
    set change-agent? FALSE
    set initiator? FALSE
    set media? FALSE
end

to setup-potential
      set status "potential"
      set color blue
end

to setup-adopter
      set status "adopter"
      if random-float 100 < adopter-agent [
        set change-agent? TRUE ;not yet used for anything
        set shape "face happy"
      ]
      set color red
end

to setup-disrupter
      set status "disrupter"
      if random-float 100 < disrupter-agent [
        set change-agent? TRUE ;not yet used for anything
        set shape "face happy"
      ]
      set color lime
end

to-report determine-kind [ probability-index ]
    let this_kind first ( first ( filter [ last ? >= probability-index ] kind-structure ) )
    print (word this_kind " : " probability-index " : " filter [ last ? >= probability-index ] kind-structure)
    report (this_kind)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; PROCESSING
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  set time time + 1

  ;People who are not currently in a conversation will:
  ; (a) think about their current status, possibly changing their mind about an adoption or disruption.  They may
  ;     return to a potential, whereby they are subject to the influences of internal or external factors.
  ;     If they are already a "technologist," they may adopt the disruption if they are already an adopter or adopt
  ;     the adoption if already a disrupter. Note that this latter case is based on "personal reflection" rather than
  ;     due to external or internal influences.
  ; (b) move a little
  ; (c) check to see if they are on a "media center" patch, in which case they may adopt after hearing the message
  ; (d) they check to see if someone is around to talk to about technologies
  ;If already paired, they update status.
  ask people [
      without-interruption [
        ;external influence is only for those whose status is not the same as the media center.
        if e-adopt and [pcolor] of patch-here = orange and status != "adopter" [
          check-external-adoption
        ]
        if e-disrupt and [pcolor] of patch-here = cyan and status != "disrupter" [
          check-external-disruption
        ]

        ;internal influence - if partner = self, mass media is in effect, so no other pairings possible
          ifelse partner = nobody [
            if status = "adopter" [rethink-adoption]
            if status = "disrupter" [rethink-disruption]
            move
            if [pcolor] of patch-here = black [ initiate ]
          ][
            if partner != self [
              interact
            ]
          ]
      ]
   ]

  update-plot-pop
  update-plot-stat
  update-plot-meet
  update-plot-type
  update-monitors
end

to check-external-adoption
  ;adoption check
    ifelse ( media? = FALSE ) [
      ;Keep track of total number of externals made
      set externals (externals + 1)
      set media? TRUE
      set partner self
      set start-time time
      set end-time round( random-exponential listening-time ) + time
      print (word time ": external influenced adoption initiated for " who " to last until " end-time)
    ][
      ifelse time >= end-time [
        set media? FALSE
        set partner nobody
        if ( time > ( start-time + time-to-adopt-external )) [
          setup-adopter
        ]
        move-away
        set num-externals (num-externals + 1)
        set total-external-time (end-time - start-time + total-external-time)
        print (word time ": " who " has finished adoption mass media session with status " status)
      ][
        print (word time ": " who " is listening to mass media presentation for adopters")
      ]
    ]
end

to check-external-disruption
  ;disruption check
    ifelse ( media? = FALSE ) [
      ;Keep track of total number of externals made
      set externals (externals + 1)
      set media? TRUE
      set partner self
      set start-time time
      set end-time round( random-exponential listening-time ) + time
      print (word time ": external influenced disruption initiated for " who " to last until " end-time)
    ][
      ifelse time >= end-time [
        set media? FALSE
        set partner nobody
        if ( time > ( start-time + time-to-disrupt-external )) [
          setup-disrupter
        ]
        move-away
        set num-externals (num-externals + 1)
        set total-external-time (end-time - start-time + total-external-time)
        print (word time ": " who " has finished disruption mass media session with status " status)
      ][
        print (word time ": " who " is listening to mass media presentation for disrupters")
      ]
    ]
end

to move
    rt random-float 360
    fd movement
    setxy pxcor pycor  ;centers on patch
end

to move-away
    rt random-float 180
    jump int(s-adopt / 100 * max-pxcor * 2) ;move away from media center
    setxy pxcor pycor ;centers on patch
end

to initiate
  ;partner with someone on own patch who it not already partnered
  ifelse (any? other people-here with [partner = nobody]) [

     ;Keep track of total number of contacts made
     set contacts (contacts + 1)

     ;may not actually strike up a conversation with this partner.  The contact rate is adjusted
     ;with the "movement" parameter and the "prob-conversation" parameter.  Note also that the
     ;user controls the length of each conversation which will also impacts contact rate.
     ifelse (random-float 100) < prob-conversation [
         converse
     ][
         print (word time ": no conversation initiated by " who)
     ]
  ][
    print (word time ": no possible parters for " who)
  ]
end

to converse
       ;Choose one of the eligible people to partner with - may want to consider other partnering strategies
       ;here - such as all on one's patch or one of patch + neighborhood
       set partner one-of other turtles-here with [partner = nobody]

       ;Set partner's attribute to me.  May want to consider a list of partners to look at memory
       ;of who one has partnered with
       set [partner] of partner self

       ;This person is the initiator, automating rendering the partner to a subordinate role
       set initiator? TRUE
       set [initiator?] of partner FALSE

       ;keep track of time conversation started
       set start-time time
       set [start-time] of partner time

       ;set time to end conversation with mean conversation length and std. deviation set to 25% of mean
       ;in multiple person interactions, can change to have some partners leave early and not adopt technology
       let conversation-end round((random-float conversation-length) + time )
       set end-time conversation-end
       set [end-time] of partner conversation-end

       ;Set patch of conversation to different color - since xcor and ycor are real numbers, patches may not exactly
       ;coincide with position of partners
       let this-color color
       set [pcolor] of patch-here this-color

       ;keep track of total number of conversations started in simulation
       set conversation-starts (conversation-starts + 1)
       print (word time ": conversation between " kind ", " status " " who " and " [kind] of partner ", " [status] of partner
             " " [who] of partner " until " end-time " at [" xcor "," ycor "]")
end

to interact
    ifelse (time <= end-time) [
       ifelse (time > start-time) [
         set [pcolor] of patch-here yellow
         print (word time ": ongoing conversation between " who " and " [who] of partner)
       ][
         print (word time ": match already established between " who " and " [who] of partner)
       ]
    ][
       ;Decide whether to adopt as conversation comes to a close
       ifelse status = "potential" and [status] of partner = "adopter" [
         if adopter-switch [
           if random-float 100 < prob-adoption [setup-adopter]
         ]
       ][
         ifelse status = "potential" and [status] of partner = "disrupter" [
           if disrupter-switch [
             if random-float 100 < prob-disruption [setup-disrupter]
           ]
         ][
              print (word time ": no action for " status " " who " in conversation with " [status] of partner " " [who] of partner)
         ]
       ]

       ;stat collection
       set num-conversations (num-conversations + 1)
       set total-interact-time (end-time - start-time + total-interact-time)
       ifelse initiator? [
          set conversation-ends (conversation-ends + 1)
          set [pcolor] of patch-here original-color
          print (word time ": initiator " who " ends conversation with " [who] of partner)
       ][
          print (word time ": recipient " who " ended conversation with initiator " [who] of partner)
       ]
       set partner nobody
       rt random-float 360
       jump 5 * movement

    ]
end

to rethink-adoption
    ifelse random-float 100 < prob-adoption-to-potential [
      setup-potential
      print (word time ": reneged adoption => " who)
    ][
      if disrupter-switch and random-float 100 < prob-adoption-to-disruption [
        setup-disrupter
        print (word time ": adoption to disruption for " who)
      ]
    ]
end

to rethink-disruption
    ifelse random-float 100 < prob-disruption-to-potential [
      setup-potential
      print (word ": reneged disruption => " who )
    ][
      if adopter-switch and random-float 100 < prob-disruption-to-adoption [
        setup-adopter
        print (word time ": disruption to adoption => " who)
      ]
    ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; PLOTTING PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup-plot
  ;Histogram - don't set limits - let model autoset
  ;set-current-plot "Meetings"
  ;set-histogram-num-bars 10
  set-current-plot "Adopter Type"
  set-histogram-num-bars 5
  update-plot-type
end

to update-plot-pop
  ;update time series
  set-current-plot "Populations"
  set-current-plot-pen "Total"
  plot count people
  set-current-plot-pen "Potentials"
  plot count people with [status = "potential"]
  set-current-plot-pen "Adopters"
  plot count people with [status = "adopter"]
  set-current-plot-pen "Disrupters"
  plot count people with [status = "disrupter"]
end

to update-plot-stat
  ;contacts, conversations and meetings
  set-current-plot "Statistics"
  set-current-plot-pen "Contacts"
  plot contacts / time
  set-current-plot-pen "Starts"
  plot (conversation-starts / time)
  set-current-plot-pen "Ends"
  plot (conversation-ends / time)
  set-current-plot-pen "Externals"
  plot externals / time
end

to update-plot-meet
  ;update meetings histogram
  set-current-plot "Meetings"
  set-current-plot-pen "Internal"
  histogram [num-conversations] of people
  set-current-plot-pen "External"
  histogram [num-externals] of people
end

to update-plot-type
  ;update adopter type histogram
  set-current-plot "Adopter Type"
  set-current-plot-pen "kind"
  histogram [kind] of people
end

;;;
;;; MONITOR PROCEDURES
;;;

to update-monitors
  ;reset counters for each year
  set contacts-per-time (contacts - old-contacts)
  set starts-per-time (conversation-starts - old-conversation-starts)
  set ends-per-time (conversation-ends - old-conversation-ends)
  set active-conversations (conversation-starts - conversation-ends)
  set old-contacts contacts
  set old-conversation-starts conversation-starts
  set old-conversation-ends conversation-ends
end

@#$#@#$#@
GRAPHICS-WINDOW
293
10
950
637
25
23
12.7
1
10
1
1
1
0
1
1
1
-25
25
-23
23
1
1

CC-WINDOW
5
761
1256
856
Command Center
0

MONITOR
1161
10
1247
55
Simulated Time
time
0
1
11

BUTTON
960
10
1023
56
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
1090
10
1158
56
Start-Stop
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
1025
10
1088
56
Step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
7
165
272
198
initial-potentials
initial-potentials
0
1000
95
1
1
people
HORIZONTAL

SLIDER
10
337
275
370
prob-conversation
prob-conversation
0
100
100
0.1
1
percent
HORIZONTAL

SLIDER
10
373
275
406
conversation-length
conversation-length
0
100
10
1
1
time units
HORIZONTAL

PLOT
961
63
1246
197
Populations
Simulated Time
Number of People
0.0
10.0
0.0
10.0
true
true
PENS
"Potentials" 1.0 0 -13345367 true
"Adopters" 1.0 0 -2674135 true
"Disrupters" 1.0 0 -13840069 true
"Total" 1.0 0 -16777216 true

PLOT
960
476
1245
632
Meetings
Number of Meetings
Frequency
0.0
100.0
0.0
10.0
true
true
PENS
"Internal" 1.0 1 -13791810 true
"External" 1.0 1 -5825686 true

SLIDER
7
202
273
235
initial-adopters
initial-adopters
0
100
5
1
1
people
HORIZONTAL

SLIDER
9
240
274
273
initial-disrupters
initial-disrupters
0
100
0
1
1
people
HORIZONTAL

TEXTBOX
78
146
228
164
Demographic Parameters
11

TEXTBOX
71
412
221
430
Internal Influence Parameters
11

SLIDER
9
34
269
67
seed
seed
0
100000000
52244898
1
1
NIL
HORIZONTAL

SLIDER
12
433
277
466
prob-adoption
prob-adoption
0
100
33.3
0.1
1
percent
HORIZONTAL

SLIDER
12
545
278
578
prob-disruption
prob-disruption
0
100
0
0.1
1
percent
HORIZONTAL

SLIDER
10
300
276
333
movement
movement
0
10
1
1
1
patches per move
HORIZONTAL

TEXTBOX
87
280
237
298
Contact Parameters
11

SLIDER
11
508
278
541
prob-adoption-to-disruption
prob-adoption-to-disruption
0
5
0
0.1
1
percent
HORIZONTAL

SLIDER
11
471
277
504
prob-adoption-to-potential
prob-adoption-to-potential
0
5
0
0.1
1
percent
HORIZONTAL

SWITCH
10
72
139
105
adopter-switch
adopter-switch
0
1
-1000

SWITCH
10
107
139
140
disrupter-switch
disrupter-switch
1
1
-1000

SLIDER
11
581
278
614
prob-disruption-to-potential
prob-disruption-to-potential
0
5
0
0.1
1
percent
HORIZONTAL

SLIDER
10
617
277
650
prob-disruption-to-adoption
prob-disruption-to-adoption
0
5
0
0.1
1
percent
HORIZONTAL

TEXTBOX
471
651
693
688
External Influence Parameters
11

PLOT
960
199
1247
334
Statistics
Simulated Time
Number of People
0.0
10.0
0.0
10.0
true
true
PENS
"Contacts" 1.0 0 -16777216 true
"Starts" 1.0 0 -13791810 true
"Ends" 1.0 0 -5825686 true
"Externals" 1.0 0 -955883 true

MONITOR
960
636
1095
681
Contacts  per Time Period
contacts-per-time
0
1
11

MONITOR
1105
636
1244
681
Starts perTime Period
starts-per-time
0
1
11

MONITOR
960
692
1096
737
Ends per Time Period
ends-per-time
0
1
11

MONITOR
1104
691
1245
736
No. Active Conversations
active-conversations
0
1
11

SLIDER
9
675
280
708
adopter-agent
adopter-agent
0
100
0
0.1
1
percent
HORIZONTAL

SLIDER
10
712
280
745
disrupter-agent
disrupter-agent
0
100
0
0.1
1
percent
HORIZONTAL

SLIDER
398
713
491
746
x-disrupt
x-disrupt
-100
100
0
1
1
NIL
HORIZONTAL

SLIDER
397
679
489
712
x-adopt
x-adopt
-100
100
0
1
1
NIL
HORIZONTAL

SLIDER
493
714
585
747
y-disrupt
y-disrupt
-100
100
0
1
1
NIL
HORIZONTAL

SLIDER
491
679
585
712
y-adopt
y-adopt
-100
100
0
1
1
NIL
HORIZONTAL

TEXTBOX
95
656
245
674
Change Agents (TBD)
11

SLIDER
685
677
949
710
time-to-adopt-external
time-to-adopt-external
0
50
0
1
1
time units
HORIZONTAL

SLIDER
686
713
950
746
time-to-disrupt-external
time-to-disrupt-external
0
50
10
1
1
time units
HORIZONTAL

SLIDER
587
678
679
711
s-adopt
s-adopt
0
100
0
1
1
NIL
HORIZONTAL

SLIDER
587
714
679
747
s-disrupt
s-disrupt
0
100
25
1
1
NIL
HORIZONTAL

SWITCH
143
73
269
106
e-adopt
e-adopt
1
1
-1000

SWITCH
143
107
269
140
e-disrupt
e-disrupt
1
1
-1000

SLIDER
686
642
950
675
listening-time
listening-time
0
50
8
1
1
time units
HORIZONTAL

PLOT
960
335
1245
476
Adopter Type
Innovator | EA | EM | LM | Laggard
Frequency
0.0
6.0
0.0
10.0
true
false
PENS
"kind" 1.0 1 -13345367 true

TEXTBOX
95
10
245
28
Model Parameters
11

@#$#@#$#@
VERSION
-------
$Id: Innovation.nlogo 37529 2008-01-03 20:38:02Z craig $


WHAT IS IT?
-----------
This is a model of the diffusion of innovations, part of an overall Innovation Process that covers invention, R&D, production of innovations, dissemination of those innovations, and various life cycle activities through obsolescence and retirement of old technologies. It focuses on the diffusion and adoption of new technologies based on "internal influences" (e.g., word-of-mouth) and "external influences" (e.g., mass media).  Individuals are divided into three groups: "Potentials" who have not yet adopted any new technologies, "Adopters" who are using a new technology, and "Disrupters" who are using an even newer technology than "Adopters."

Individuals move through the state space, where they meet and interact with other individuals.  If a "Potential" interacts with an "Adopter," he or she may become an "Adopter" based on probabilities selected at runtime.  The same holds true for "Potentials" interacting with "Disrupters."  Other rules control the interaction of "Adopters" and "Disrupters."  There are also parameters for "Adopters" to "renege" and return to "Potentials" or to accept the even newer "Disrupter" technology and discard their "Adopter" innovation.  Similar parameters are available for "Disrupters."

Mass media impacts may also be studied.  The impact of mass media is based on the length of time a user spends in a particular region (colored orange for the "Adopter" technology and cyan for the "Disrupter" technology).

Toggle switches are provided to control each simulation so that a study may focus on internal or external influences of one technology in isolation.


HOW IT WORKS
------------
The model can be divided into three parts, each of which is described below:

(1) Individuals - The modeler selects an initial number of "Potentials," "Adopters" and "Disrupters" from the "Demographic Parameters" section of the interface.  When the "setup" button is pressed, "Potentials" are created as blue figures.  "Adopters" are red figures, but they will only appear in the model if the "adopter-switch" is ON.  "Disrupters" will be green, but again, they only appear if the "disrupter-switch" is ON.

When the simulation begins, individuals move around the display, occasionally (depending on parameters) meeting other individuals of the same or different "type."  They may engage in conversation, which may or may not lead to one of the individuals changing "type" (e.g., from a "Potential" to an "Adopter.")  Or, they may change "type" based on the length of time they have listened to a mass media presentation.

(2) Internal Influence Diffusion ("word-of-mouth")

People who are not currently in a "conversation" will perform a number of steps:

 (a) Think about their current status, possibly changing their mind after accepting a new technology:  they may return to a "Potential," whereby they are subject to the influences of internal or external factors.  If they are a technologist ("Adopter" or "Disrupter"), they may become a "Disrupter" if they are currently an "Adopter" or an "Adopter" if currently a "Disrupter."  Note that these changes are based on "personal reflection" rather than due to external or internal influences.  These changes are based on slider values "prob-adoption-to-potential," "prob-adoption-to-disruption," "prob-disruption-to-potential" and "prob-disruption-to-adoption."

 (b) Move a little based on the "movement" slider set in the interface.

 (c) Check to see if they are on a "media center" patch, in which case they may accept a new technology after hearing a "mass media presentation." (see comments below under "External Diffusion.")

 (d) Finally, they check to see if someone is around to talk to about technologies.  The "movement" parameter" and "prob-conversation" sliders help determine how often conversations occur - just meeting someone does not guarantee that a conversation about technology occurs.  If a conversation does occur, one individual is the initiator and the other the recipient of information.

If an individual is already paired, he or she will update his or her status.  Whether to accept the new technology is based on the slider probability for each technology.  This is only checked at the end of the conversation, not once per time unit.  These sliders are "prob-adoption" and "prob-disruption."

NOTE: The "Change Agent" sliders are not yet used.  For now, every "Adopter" and "Disrupter" is set as a Change Agent, but this has no impact.  The intent is to have a subset of each technology group be "Change Agents" that can "grab" individuals in the surrounding patches and exert influence on a number of other individuals at once.  In the current model, conversations only occur between pairs, and the pairs must be on the same patch.

(3) External Diffusion - The user chooses an (x,y) coordinate for the "Adopter" and "Disrupter" mass media centers, noting that (0,0) is the center of the diagram, with values ranging from -100 to 100 for each axis.  The user then chooses a "half-width" from this point, establishing a square "media center" from (x-s,y-s) to (x+s,y+s).  Whether the media center is active or not is set by an ON/OFF toggle switch.  The media centers should not overlap in space.

If a media center is active, sliders are set for "listening time," the mean of an exponential distribution used to determine how long the user will listen to the message in the media center.  In addition, the modeler will set a "time to accept new technology" for the "Adopter Media Center" and a separate time for the "Disrupter Media Center."

When a person is in a media center, they do not pair with anyone; they just listen to the message for an amount of time set by EXPON("listening_time").  When they are about to leave the media center, the total time spent is compared to the slider values for each technology.  The individual's status will change to that technology if the time spent in the media center is greater than the time set by the modeler.  Nothing happens to an individual of the same type as the media center, but individuals may switch types in the current model.  When they are done, they "jump" to the edge of the media center and begin to interact with others again.


HOW TO USE IT
-------------
Buttons

	(1) Setup - uses startup parameters to establish a set of individuals of different technology classes, along with media centers.
	(2) Step - can move one step at a time - very slow at beginning of model.  Can run for awhile, stop, and then step.
	(3) Start-Stop - toggles between "GO" state and "STOP" state.

Sliders

	(1) Model Parameters

	    (a) Seed - change the initial random number seed to run a new "experiment" with the same inputs
	    (b) adopter-switch - ON/OFF toggle switch for "Adopter" technology and its diffusion through word-of-mouth
	    (c) disruption-switch - ON/OFF toggle switch for "Disrupter" technology
	    (d) e-adopt - ON/OFF switch for mass media external influences for "Adopter" technology
	    (e) e-disrupt - ON/OFF toggle switch for mass media messages of "Disrupter" technology

	(2) Demographic Parameters

	    (a) initial-potentials - starting number of potential adopters
	    (b) initial-adopters - starting number of "Adopters" who already use a new technology.  These, along with "Disrupters," constitute "Innovators," the most tech-saavy user group using Rogers' parlance.
	    (c) initial-disrupters - initial number of "Disrupters" who already use technology even newer than "Adopters."  As noted, these individuals are part of the "Innovators" group.

	(3) Contact Parameters

	    (a) movement - this parameter determines how many patches an individual will move in a single "step."
	    (b) prob-conversation - Just because individuals meet, it doesn't mean they will exchange information.  The combination of this parameter (a percentage) and the movement parameter can be used to calibrate the model against analytical or system dynamics models of diffusion.
	    (c) conversation-length - This parameter decides how long people are "tied up" in a converstation, but it does not affect the probability of accepting a new technology.

	(4) Internal Influence Parameters

	    (a) prob-adoption - At the end of a conversation, there is a probability that a user will adopt the new technology - assuming that the individuals in question are supporters of different levels of technology
	    (b) prob-adoption-to-potential - "Adopters" can renege and return to being "Potentials"
	    (c) prob-adoption-to-disruption - "Adopters" may decide to accept a newer technology
	    (d) prob-disruption - probability of accepting the latest technology
	    (e) prob-disruption-to-potential - renege case for "Disrupters"
	    (f) prob-disruption-to-adoption - "Disrupters" may decide the earlier innovation is better after all

	(5) Change Agents - not yet used

	    (a) adapter-agent - percentage of total number of "Adopters" who can act as Change Agents, impacting multiple "Potentials" at once
	    (b) disrupter-agent - similar percentage for "Disrupters"

	(6) External Influence Parameters

	    (a) x-adopt - x-coordiante of center of square media center for "Adopters"
	    (b) y-adopt - y-coordinate of center of "Adopter" media center
	    (c) s-adopt - "Adopter" media center ranges from (x-s,y-s) to (x+s,y-s) to (x+s,y+s) to (x-s,y+s)
	    (d) x-disrupt - "Disrupter" media center x-coordinate
	    (e) y-disrupt - "Disrupter" media center y-coordinate
	    (f) s-disrupt - "Disrupter" media center ranges from (x-s,y-s) to (x+s,y-s) to (x+s,y+s) to (x-s,y+s)

Outputs
	(1) Simulated Time - keeps track of time steps
	(2) Populations - total number of individuals of each type
	(3) Statistics - number of contacts made, number of started conversations and number of finished conversations, number of individuals who listen in media centers.
	(4) Adopter Type - histogram of types of users.
	(5) Meetings - histograph of two-person conversations and media center meetings
	(6) Counters for the number of contacts made in a time period, how many started in that time period, how many ended, and number of active conversations.


THINGS TO NOTICE
----------------
Best idea is to run one technology at a time ("Adopter") in word-of-mouth mode, followed by external mode, followed by a mixed mode.  These results can be compared to analytical models discussed in the REFERENCES section below.


THINGS TO TRY
-------------
Note change in rate of technology adoption based on parameters.  The "expected" behavior is a logistic curve for a single technology introduced by word-of-mouth.  A strictly "mass media" introduction will spread more rapidly, shooting up to the maximum after a small delay (depending upon parameter settings).

Adding a second technology after a short delay leads to all sorts of interesting behavior.  In some cases, the newer technology will not be able to gain a foothold.  If reneging is added, or if "Adopters" can switch to the "Disrupter" technology, the technology curves become quite complex.  According to theory, at least, a "Disrupter" technology often needs to hit a specific niche in order to gain a foothold once an "Adopter" technology has saturated the market (e.g., "WII" vs Playstation 3 vs. Xbox 360 marketing strategies).


EXTENDING THE MODEL
-------------------
(1) Change agents - Rogers (see references) talks about the importance of change agents to facilitate adoption of new technologies or ideas.  In the current model, two individuals will meet; adoption is based on parameters such as length of time of the conversation.  However, some users could be initialized as change agents, in which case they might require much less time to convince certain categories of users.

(2) Innovation Group - Innovators, Early Adopters, Early Majority, Late Majority, Laggards.

(3) Internet - The classic diffusion models differentiate individual contacts from mass media.  The Internet, however, is really a combination of the two, providing "intimate" conversations with a single user, but marketing to a wide audience.


NETLOGO FEATURES
----------------

The most interesting feature has to do with  setting the "adoption category" of individuals.  Some folks are "early adopters" and will try anything new.  Some will never adopt anything new, and others are spread out into other groups.  Rogers (see references below) discusses a normal distribution with 5 categories such as 1 & 2 (the early adopters) = 5 (adopt with a lot of resistance) and two middle categories, 3 & 4.  This is referred to as the "kind-structure" in the model. Category 1 is set by the number of "adopters" + "disrupters" - those who already have the new innovation at the start of the simulation.  To determine the rest of the population, an array is set up in the initialization part of the model:

=============
set kind-structure [ [ 2 12 ] [ 3 47 ] [ 4 81 ] [ 5 100 ] ]
=============

The first entry is the category type; the second is a probability.  This is used in the following reporter routine:

=============
to-report determine-kind [ probability-index ]
    let this_kind first ( first ( filter [ last ? >= probability-index ] kind-structure ) )
    print this_kind+":"+probability-index+":"+filter [last ? >= probability-index] kind-structure
    report (this_kind)
end
=============

the "let" statement was graciously explained to me through the community web pages.  It took awhile to convince myself it was working, but it is a very useful structure for this and similar demographic modeling in which the inputs are often categories assigned probabilities.  This routine is called during setup in this model to determine the kind of adopter:

=============
to setup-people

  cct-people initial-potentials [
      set total-population initial-potentials + total-population
      setup-people-parms
      setup-potential
      set kind determine-kind ( random 100 )
    ]

end
=============

A good use of the "kind" attribute is to assign different adoption rates based on the kind of person under consideration.  This will hopefully be added in the future.

RELATED MODELS
--------------
This "Innovation Model" took many of the two-person interaction concepts from the "AIDS Model" in the Netlogo Library.  In addition, a Vensim model was created to help verify the results of this agent-based version.


CREDITS AND REFERENCES
----------------------
The background for this work comes from a small monograph put out by Sage Publications:

(1) Mahajan, Vijay and Robert A. Peterson, "MODELS FOR INNOVATION DIFFUSION," Quantitative Applications in the Social Sciences, Sage Publications, 88 pages.

F. Bass is given credit for one of the earliest models of the diffusion process:

(2) Bass, F. M. (1969). "A new product growth model for consumer durables". Management Science, 15, 215-227.

The "Bible" of Innovation is Rogers work, now in its 5th edition:

(3) Rogers, Everett M., Diffusion of Innovations, 5th Edition, Free Press, 512 pp.

This model is actually a small portion of an ongoing effort to understand the "Innovation Process" from the invention of ideas and technologies to their development, diffusion, use and obsolescence.  The overall model is being developed through System Dynamics, but agent-based modeling has been very helpful in clarifying ideas concerning the competition and spread of technologies.

=================================
Copyright 2007 Michael L. Samuels
Last updated: 01/08/2007
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 109 6 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
Circle -1184463 true false 126 33 14
Circle -1184463 true false 156 34 13
Line -1184463 false 158 70 167 62
Polygon -1184463 true false 141 63 148 40 154 63
Rectangle -1184463 true false 135 70 160 75
Line -1184463 false 134 70 126 61
Polygon -1184463 true false 58 162 34 196 37 190
Rectangle -16777216 true false 30 180 45 195
Rectangle -16777216 true false 45 180 60 165
Rectangle -16777216 true false 60 180 45 165
Rectangle -16777216 true false 45 165 60 180

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.0pre7
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
