<?xml version="1.0" encoding="utf-8"?>
<model version="NetLogo 7.0.0" snapToGrid="true">
  <code><![CDATA[globals [
  resistance-1
  resistance-2
  therapies ; Array of the therapeutic options
  xloc
  yloc
  Months
  last-radiology
  low-threshold
  medium-threshold
]

breed [ cells cell ]

cells-own [
  resistance-type ; 0 for no resistance, 1 for resistant to therapy 1, 2 for resistant to therapy 2, and 3 for resitant to both
]

to setup
  clear-all
  ifelse seed = 0 [
    random-seed new-seed
  ] [
    random-seed seed
  ]
  
  ifelse god-mode [ 
    display 
  ] [
    no-display
  ]
  set-default-shape cells "circle"
  set xloc 0
  set yloc 0
  set Months 0
  set last-radiology -2
  set low-threshold 0.33
  set medium-threshold 0.67
  
  if (num-cells-that-die >= num-cells-that-compete) [
    print "The number of cells that die cannot be greater or equal to the number of cells that compete."
    stop
  ]
  ;; Set up the therapeutic options based on num-treatment-options (I could make this the maximum list and then only use the first N)
  set therapies [ "Paclitaxel" "Carboplatin" "Radiation" "Tamoxifen" "Herceptin" "Pembrolizumab"]
  ;; Randomly choose which two therapies the tumor is sensitive to
  choose-sensitivities
  
  ;; Initialize the tumor with num-starting-cells, proportion-resistant of which should be one of the 3 forms of resistance
  generate-starting-tumor
  
  ;; The cancer has just been detected by radiology
  radiology-scan
end



;; Chooses two different therapies the tumor may be sensitive to
to choose-sensitivities
  let res-1 random (length therapies)
  if god-mode [print (word "Sensitive to " item res-1 therapies)]
  let res-2 random (length therapies)
  while [res-1 = res-2] [
    set res-2 random (length therapies)
  ]
  if god-mode [ print (word "And sensitive to " item res-2 therapies) ]
  set resistance-1 (item res-1 therapies)
  set resistance-2 (item res-2 therapies)
end



to generate-starting-tumor
  create-cells num-starting-cells [
    set color [255 100 100 128]
    set size (1 + random-float 3) ; easier to see
    setxy xloc yloc ; get the location of the last cell
    ; Place this cell near the last one
    rt random 360
    fd random-float 3
    set xloc xcor 
    set yloc ycor
    ; Check if this is a resistant cell
    ifelse (random-float 1) < proportion-resistant [
      set resistance-type ((random 3) + 1) ; equal probability of the 3 different resistance types
    ] [
      set resistance-type 0 ; sensitive to both therapies
    ]
  ]
end

to radiology-scan 
  ifelse last-radiology < (Months - 1) [
    ifelse (count cells) >= num-cells-detectable-by-radiology [
      display
      let tumor-size ((count cells) / 10)
      print (word "Radiology estimates a tumor size of " tumor-size " cubic centimeters")
      set last-radiology Months
      no-display
    ] [
      ;no-display
      update-plots
      print (word "Radiology did not detect any tumor")
      set last-radiology Months
    ]
  ] [
    print (word "You last ordered a CT scan on month " last-radiology ". Insurance will not pay for a CT scan this soon.")
    ;no-display
  ]
end


to treat
  if Treatment-1 != "None" [ print (word "Month " Months ": Treating with " Treatment-1 " at " Dose-1 " dose") ]
  if Treatment-2 != "None" [ print (word "Month " Months ": Treating with " Treatment-2 " at " Dose-2 " dose") ]
  set Months Months + 1
  ask cells [
    ;print (word "Resistance type " resistance-type)
    ; If resistance-1 = Treatment-1 or resistance-1 = Treatment-2 then the cell is sensitive if resistance-type is 0 or 2
    if (resistance-1 = Treatment-1) or (resistance-1 = Treatment-2) [
      if (resistance-type = 0) or (resistance-type = 2) [
        if (Dose-1 = "High") or ((Dose-1 = "Low") and (random 2 = 0)) [  ; Low dose has a 50% chance of killing a sensitive cell
          ;print "Dying"
          die
        ]
      ]
    ]
    ; If resistance-2 = Treatment-1 or resistance-2 = Treatment-2 then the cell is sensitive if resistance-type is 0 or 1
    if (resistance-2 = Treatment-1) or (resistance-2 = Treatment-2) [
      if (resistance-type = 0) or (resistance-type = 1) [
        if (Dose-1 = "High") or ((Dose-1 = "Low") and (random 2 = 0)) [  ; Low dose has a 50% chance of killing a sensitive cell
          die
        ]
      ]
    ]
  ]
  cancer-cells-proliferate
  cancer-cells-compete
  
  ; Losing condition
  let tumor-size ((count cells) / 10)
  if tumor-size > size-of-lethal-cancer [
    print (word "The cancer exceeded the lethal size. It grew to " tumor-size " cubic centimeters after " Months " months since diagnosis.")
  ]
  
  ; Winning condition
  if Months >= months-to-survive [
    print (word "Success! You have kept the patient alive for " Months " months.")
  ]
end

to cancer-cells-proliferate
  ;let num-muts 0
  ask cells [
    hatch 1 [
      ;print "Dividing!"
      set size (1 + random-float 3)
      rt random 360
      fd random-float 3
      if (random-float 1) < mutation-rate [
        ;set num-muts (num-muts + 1)
        (ifelse
          resistance-type = 0 [
            set resistance-type (1 + (random 2)) ; randomly choose a single resistance
          ]
          resistance-type = 1 [
            ifelse allow-back-mutations and (random 2) = 1 [
              set resistance-type 0
            ] [
              set resistance-type 3 ; become doubly resistant
            ]
          ]
          resistance-type = 2 [
            ifelse allow-back-mutations and (random 2) = 1 [
              set resistance-type 0
            ] [
              set resistance-type 3  ; become doubly resistant
            ]
          ]
          [ ; resistance-type = 3 ; doubly resistant
            if allow-back-mutations [
             set resistance-type (1 + (random 2)) ; randomly choose a single resistance             
            ]
          ] ; else don't do anything
        )
      ]
    ]
  ]
  ;print (word "There were " num-muts " mutations this month.")
end

; Choose groups of num-cells-that-compete and kill off the weakest num-cells-that-die
to cancer-cells-compete
  let cell-list shuffle (sort cells)
  let n length cell-list
  ;print (word "There are " n " cells before competition.")
  ; go through in groups of num-cells-that-compete
  foreach (range 0 n num-cells-that-compete) [
    i ->
    let group sublist cell-list i (min list (i + num-cells-that-compete) n)
    ; Only allow competition if you have a full set of competitors
    if (length group) = num-cells-that-compete [
      let competing-cells turtle-set group
      ;show [resistance-type] of competing-cells
      let weakest max-n-of num-cells-that-die competing-cells [resistance-type]
      ;show (word "Weakest turtles is #" [who] of weakest " with resistance type " [resistance-type] of weakest)
      ask weakest [ die ]
    ]
  ]
  ;print (word "There are " (count cells) " cells after competition.")
end

to run-biomarker-test
  let frequency-sensitive 0
  ifelse (count cells) >= num-cells-detectable-by-radiology [
    (ifelse
      ; The chemo and radiation therapies are more complicated. They work best on highly proliferating cells, but seeing
      ; proliferation does not tell you which of the therapies it will be sensitive to.
      Diagnostic-Test = "Proliferation" [
        if (resistance-1 = "Paclitaxel" or resistance-1 = "Carboplatin" or resistance-1 = "Radiation") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 2])) / (count cells)
        ]
        if (resistance-2 = "Paclitaxel" or resistance-2 = "Carboplatin" or resistance-2 = "Radiation") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 1])) / (count cells)
        ]         
      ]
      Diagnostic-Test = "Estrogen receptor" [
        if (resistance-1 = "Tamoxifen") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 2])) / (count cells)
        ]
        if (resistance-2 = "Tamoxifen") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 1])) / (count cells)
        ]         
      ]
      Diagnostic-Test = "HER2" [
        if (resistance-1 = "Herceptin") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 2])) / (count cells)
        ]
        if (resistance-2 = "Herceptin") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 1])) / (count cells)
        ]      
      ]
      Diagnostic-Test = "PD-L1" [
        if (resistance-1 = "Pembrolizumab") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 2])) / (count cells)
        ]
        if (resistance-2 = "Pembrolizumab") [
          set frequency-sensitive ((count cells with [resistance-type = 0]) + (count cells with [resistance-type = 1])) / (count cells)
        ]
    ] [])
  
    ; Return test results
    (ifelse
      frequency-sensitive = 0 [
        print (word "The " Diagnostic-Test " test was negative.")
      ]
      frequency-sensitive < low-threshold [
        print (word "The tumor is weakly positive  for " Diagnostic-Test ".")
      ]
      frequency-sensitive < medium-threshold [
        print (word "The tumor is moderately positive for " Diagnostic-Test ".")
      ]
      [
        print (word "The tumor is strongly positive for " Diagnostic-Test ".")
    ])
  ] [
    print "You can't find any tumor to biopsy for the test."
  ]
end
]]></code>
  <widgets>
    <view x="370" wrappingAllowedX="false" y="55" frameRate="30.0" minPycor="-16" height="434" showTickCounter="true" patchSize="13.0" fontSize="10" wrappingAllowedY="false" width="434" tickCounterLabel="ticks" maxPycor="16" updateMode="1" maxPxcor="16" minPxcor="-16"></view>
    <note x="840" y="10" backgroundDark="0" fontSize="14" width="150" markdown="false" height="25" textColorDark="-1" textColorLight="-16777216" backgroundLight="0">Game Parameters</note>
    <slider x="840" step="1" y="35" max="100" width="250" display="num-starting-cells" height="50" min="0" direction="Horizontal" default="10.0" variable="num-starting-cells"></slider>
    <slider x="840" step="0.01" y="89" max="1" width="250" display="proportion-resistant" height="50" min="0" direction="Horizontal" default="0.6" variable="proportion-resistant"></slider>
    <slider x="840" step="1" y="143" max="20" width="250" display="num-cells-that-compete" height="50" min="0" direction="Horizontal" default="8.0" variable="num-cells-that-compete"></slider>
    <slider x="840" step="1" y="197" max="10" width="250" display="num-cells-that-die" height="50" min="0" direction="Horizontal" default="2.0" variable="num-cells-that-die"></slider>
    <slider x="840" step="1" y="251" max="20" width="250" display="num-cells-detectable-by-radiology" height="50" min="0" direction="Horizontal" default="10.0" variable="num-cells-detectable-by-radiology"></slider>
    <slider x="840" step="0.01" y="305" max="1" width="250" display="mutation-rate" height="50" min="0" direction="Horizontal" default="0.25" variable="mutation-rate"></slider>
    <slider x="840" step="1" y="359" max="100" width="250" display="size-of-lethal-cancer" height="50" min="0" direction="Horizontal" default="5.0" variable="size-of-lethal-cancer" units="cc"></slider>
    <slider x="840" step="1" y="413" max="100" width="250" display="months-to-survive" height="50" min="0" direction="Horizontal" default="24.0" variable="months-to-survive"></slider>
    <button x="25" y="15" height="40" disableUntilTicks="false" forever="false" kind="Observer" width="55" display="Setup">setup</button>
    <button x="90" y="15" height="40" disableUntilTicks="false" forever="false" kind="Observer" width="110" display="Start Treatment">treat</button>
    <chooser x="25" y="70" height="60" variable="Diagnostic-Test" current="2" width="175" display="Diagnostic-Test">
      <choice type="string" value="Proliferation"></choice>
      <choice type="string" value="Estrogen receptor"></choice>
      <choice type="string" value="HER2"></choice>
      <choice type="string" value="PD-L1"></choice>
    </chooser>
    <button x="210" y="70" height="60" disableUntilTicks="false" forever="false" kind="Observer" width="80" display="Order Test">run-biomarker-test</button>
    <button x="25" y="135" height="55" disableUntilTicks="false" forever="false" kind="Observer" width="70" display="CT Scan">radiology-scan</button>
    <chooser x="25" y="195" height="60" variable="Treatment-1" current="0" width="210" display="Treatment-1">
      <choice type="string" value="None"></choice>
      <choice type="string" value="Paclitaxel"></choice>
      <choice type="string" value="Carboplatin"></choice>
      <choice type="string" value="Radiation"></choice>
      <choice type="string" value="Tamoxifen"></choice>
      <choice type="string" value="Herceptin"></choice>
      <choice type="string" value="Pembrolizumab"></choice>
    </chooser>
    <chooser x="25" y="265" height="60" variable="Treatment-2" current="0" width="210" display="Treatment-2">
      <choice type="string" value="None"></choice>
      <choice type="string" value="Paclitaxel"></choice>
      <choice type="string" value="Carboplatin"></choice>
      <choice type="string" value="Radiation"></choice>
      <choice type="string" value="Tamoxifen"></choice>
      <choice type="string" value="Herceptin"></choice>
      <choice type="string" value="Pembrolizumab"></choice>
    </chooser>
    <chooser x="240" y="195" height="60" variable="Dose-1" current="0" width="100" display="Dose-1">
      <choice type="string" value="High"></choice>
      <choice type="string" value="Low"></choice>
    </chooser>
    <chooser x="240" y="265" height="60" variable="Dose-2" current="0" width="100" display="Dose-2">
      <choice type="string" value="High"></choice>
      <choice type="string" value="Low"></choice>
    </chooser>
    <monitor x="25" precision="17" y="340" height="60" fontSize="11" width="100" display="Sensitives">ifelse-value god-mode [ 
  count cells with [resistance-type = 0] 
  ] [
  ""
  ]</monitor>
    <monitor x="130" precision="17" y="340" height="60" fontSize="11" width="100" display="T1 resistant">ifelse-value god-mode [ 
  count cells with [resistance-type = 1] 
  ] [
  ""
  ]</monitor>
    <monitor x="25" precision="17" y="405" height="60" fontSize="11" width="100" display="T2 resistant">ifelse-value god-mode [ 
  count cells with [resistance-type = 2] 
  ] [
  ""
  ]</monitor>
    <monitor x="130" precision="17" y="405" height="60" fontSize="11" width="100" display="Doubly resistant">ifelse-value god-mode [ 
  count cells with [resistance-type = 3] 
  ] [
  ""
  ]</monitor>
    <monitor x="220" precision="17" y="5" height="60" fontSize="11" width="100" display="Month">Months</monitor>
    <switch x="240" y="340" height="40" on="false" variable="god-mode" width="90" display="god-mode"></switch>
    <note x="435" y="15" backgroundDark="0" fontSize="24" width="295" markdown="false" height="40" textColorDark="-1" textColorLight="-16777216" backgroundLight="0">Treat Cancer Like a Boss!</note>
    <switch x="840" y="465" height="40" on="false" variable="allow-back-mutations" width="155" display="allow-back-mutations"></switch>
    <input x="840" multiline="false" y="510" height="60" variable="seed" type="number" width="250">0.0</input>
  </widgets>
  <info><![CDATA[## WHAT IS IT?

This simulation replicates the rules of a card game we are developing to simulate an oncologist and patient treating her metastatic (breast) cancer. It illustrates a number of features of real cancer therapy: 

  1. Cancers are often unresponsive to some of the available therapies, but this can be different from patient to patient.
  2. Cancers are heterogenous, often with some (but not all) cells that are resistant to a therapy.
  3. Cancer cells can be resistant to multiple therapies. 
  4. Cancers mutate and evolve in response to our therapies.
  5. We often don't know if a therapy is working until well after the application of the therapy.
  6. If a cancer becomes undetectable, we often don't know if we have achieved a cure, or if it is just too small to detect.

## HOW IT WORKS (The rules of the game)

The tumor starts (is detected at) a certain number of cells (a game parameter). It is only sensitive to two of the possible therapies, but which two is initially unclear. In addition, it may harbor cells that are resistant to one or both of the remaining therapies.

The potential breast cancer therapies include:

  1. Paclitaxel (a chemotherapy)
  2. Carboplatin (a chemotherapy)
  3. Radiation therapy
  4. Tamoxifen (a hormone therapy targeting ER+ breast cancer cells)
  5. Herceptin (a targeted therapy targeting HER2+ breast cancer cells)
  6. Pembrolizumab (an immunotherapy that targets cells using PD-L1 to evade the immune system)

Every turn (month) the oncologist-patient team decides on a therapy. They can use 0, 1 or 2 therapies, and each therapy may be given at high dose (killing all sensitive cells) or low dose (killing sensitive cells with 50% probability). A CT scan may be employed once every 2 months (a typical schedule that health insurance will pay for). That just tells the size of the tumor (in cubic centimeters). There is a lower limit to the sensitivity of the CT scan (a game parameter), so if the CT scan is negative, it is unclear if you have achieved a cure or if the tumor is just too small at the moment to be detected. The results of the CT scan is given in cubic centimeters (cc), where a single cell is represented (unrealistically) as 0.1 cc.

At any time, the oncologist-patient team can request any of the tests that give information about what proportion of the tumor may be sensitive to the various therapies. If the tumor is too small to be detectable, you won't be able to find it to biopsy it for the tests. The posible tests include:

  1. Proliferation (sensitive to one of the threapies: Paclitaxel, Carboplatin or radiation)
  2. Estrogen receptor (ER+ cells are sensitive to Tamoxifen)
  3. HER2 (HER2+ cells are sensitive to Herceptin)
  4. PD-L1 (PD-L1+ cells are sensitive to Pembrolizumab)

The tests return negative, low (<33% of cells are positive), moderate (>33% but <67% are positive) or high (>67% are positive). That indicates the proportion of the cells that will be sensitive to the related therapy. The results of the tests, including the CT scan, are given in the Command Center window at the bottom of the interface.

Any cells that survive the therapy will duplicate. When a new cell is born, it has a chance to mutate (based on a game parameter). Sensitive cells will mutate into singly resistant cells (randomly selecting which treatment they are resistant to with equal probability). Singly resistant cells will mutate into doubly resistant cells. If the allow-back-mutations switch is on, then singly resistant cells will become fully sensitive half the time, and doubly resistant cells will become sensitive to one of the two therapies (with equal probability). 

After reproduction, groups of cells compete (exactly how many that make up a local competitive group is a game parameter). Among each group, the weakest cells die (the number that die is another game parameter). Sensitive cells are the strongest competitors, followed by cells resistant to the first therapy, followed by cells resistant to the second therapy, followed by doubly resistant cells, which are the weakest. In this way, we represent the fitness cost of resistance in their competitive ability. Which of the two therapies is designated the "first therapy" and which is the "second therapy" is not visible to the oncologist-patient team. That only matters for cell competition.

The oncologist-patient team loses if the tumor ever grows too large (the threshold is a game parameter). The oncologist-patient team wins if they keep the patient alive for a specified number of months (a game parameter). Because we are using this game to develop a card game, we have set this parameter to be relatively low (5ccs = 50 cells). You can make the game easier if you allow the tumor to grow larger before it becomes lethal.

The oncologist-patient team wins if the can keep the patient alive for some specified number of months (a game parameter). You can keep playing past the notification of winning or losing, perhaps competing to see how long you can keep the patient alive even though they still have some cancer.

The god-mode switch allows the user to see which therapies the tumor is sensitive to and the numbers of each of the four cell types. If we are able to develop tests to measure the proportion of cells that are sensitive to each of the drugs, we would be able to get something like the information revealed in god-mode. 

## HOW TO USE IT

  1. Start by hitting the setup button. This will initialize the tumor to the size that is barely detectable by a CT scan. (If seed is 0, it will start the random number generator with a random seed, but if it is not 0, it will set the random number generator from that seed so that you can get a repeatable series of random numbers which will allow you to start from the same state every time.)

  2. Run any tests you want on the cancer. The oncologist-patient team may order any number of tests every round, except CT scans can only be ordered once every 2 months. If the tumor is too small to see on a CT scan, it is also too small to biopsy, and you won't be allowed to test it. 

  3. Then choose up to two therapies to give this round and specify low or high dose for each. Therapy can be skipped by choosing "None". 

  4. Once you have selected the therapies, they press "Start treatment" to administer the therapies for a month. 

The month counter will increment by one, and then the oncologist-patient team can order a new set of tests and change therapies if you want (repeating steps 2-4 each turn). 

Results of the tests, as well as the therapies that are applied are output in the Command Center. Success or failure of your treatment plan will be output in the Command Center when it happens.

## THINGS TO NOTICE

Note that the oncologist-patient team cannot tell the effect of their last round(s) of therapy until they get the results of a CT scan. Some limited information may be gathered by the other tests every round. They can see if the tumor changes among the negative, low, moderate or high categories. They also can't ever be sure that they have achieved a cure (though if enough months have passed without any therapy and the cancer hasn't reappeared, then they can be pretty sure).

Also, if the tumor is ever reduced to only doubly resistant cells, and back mutations are not allowed, the oncologist-patient team will have no effective options left. 

The growth rate of the tumor is determined by the doubling of the surviving cells every month minus the proportion that die due to competition. If 25% die from competition, that results in 50% growth each month (e.g. 6 cells initially double to 12 cells, then a quarter die, leaving 9 cells = 50% growth from the previoius 6 cell size).

There is randomness in the model in the number of cells that are resistant, and the occurance of resistance mutations. Thus, results will vary even with identical starting tumors. You may get lucky sometimes, and achieve a cure because the tumor does not currently contain any doubly resistant cells.


## THINGS TO TRY

If you put the game into god mode, you can see what is going on in the tumor as you treat it and adjust your strategy accordingly. In the future, we hope to develop tests that will inform us about what proportion of the tumor is resistant to what therapies. You can see how much that would help.

Standard of care is generally to use high dose of drugs until the tumor starts growing under those therapies, and then switch to any remaining drugs. How does that work for you?

Can you get better results by occasionally using low dose therapies to maintain sensitive cells that can out-compete doubly resistant cells? Does skipping therapy for a month or two help to drive down the resistant cells? If you figure out which two therapies the tumor is sensitive two, can you get better results alternating between those therapies or using them at the same time?

You can make things a little easier on yourself by allowing back mutations so that sensitive cells can be regenerated from resistant cells. However, it takes awhile for them to recover, and you may not have the time if you have eliminated too many of them.

You can also set the losing condition to a larger tumor (e.g., 50 ccs, rather than 5). This gives you more leeway to leverage cell competition to your advantage and bring the tumor back under control.

## THINGS THAT ARE UNREALISTIC

> "All models are wrong, but some are useful." - George Box

This simulation is a simplication of reality, and so it makes a number of unrealistic simplifications. It also makes some assumptions that may not be true for a given real cancer. So don't take this game to your oncologist and point to it as evidence for what they should do. But do feel free to discuss the ideas in the game with them.

#### The number of cells

As for all the things that are realistic, first of all, we are simulating 10's of cells, rather than the billions to trillions of cells in a real cancer (because we also want to be able to run this model as a card game). We are also calling 1 cell = 0.1 cubic centimeter. In reality, 0.1 cubic centimeters of a tumor would include about 10 million cancer cells and ~90 million other cells. But CT scans have a hard time detecting tumors below about 1 cc, so we have made this simplification so that the CT scan results are roughly (very roughly) realistic.

The frequency of resistant cells in the simulated tumor is also unrealistic. Where one in a million cells in a real cancer might be resistant to a treatment, we are assuming that as much as 20% of the starting tumor cells are resistant to both treatments (and another 20% are resistant to each of the single treatments). But there are so many cells in a real cancer, they often harbor resistant cells before therapy starts. Similarly, probalby only one in a million cancer cell divisions would result in a mutation that makes the daughter cell resistant to a particular treatment. Here a whopping 25% of new cells have a mutation that makes them more resistant. Also, we know from the clinic that some treatments are more likely to result in a cure than other treatments, but in the game all treatments have equivalent chances of working. We have made the arbitrary rule that the tumor is sensitive to exactly two of the available treatments. In reality, some tumors are sensitive to more treatments and some tumors are sensitive to fewer treatments. Unfortunately, for many treatments, we don't have good tests that we can run that will tell us which therapies it will be sensitive to. We also know from experience that (in general), once a tumor becomes resistant to one treatment, it is harder to treat with the next treatment (a smaller fraction of tumors will respond to the "second line" treatment, and they tend to become resistant more quickly than they did to the first treatment). We don't really know why this happens, but we haven't represented that issue in this game.

#### Cell competition
We have made an extremely strong assumption about competition in the cancer. We have assumed that competition kills off a large proportion of cells (25% each cell generation). We have also assumed that therapeutic resistance comes at a cost of competitive weakness. There is some evidence for this, from a number of experiments, but we don't know how universal that cost is, and the degree of that cost. It probably varies depending on the mechanism of therapeutic resistance (there are many different ways a cell can be resistant to a therapy), and the context of the other cells it is competing with. You can actually play with this assumption. If you decrease the num-cells-that-compete and the num-cells-that-die, you reduce the amount of competition, and the game becomes even harder.

#### Treatments
Also, you should know that most cancer treatments are only given at one dose, the maximum dose that the patient can tolerate. Those are only lowered if the drugs are causing unacceptable toxicity. We aren't even simulating toxicty (the harmful effects of therapies on your healthy tissues). However, we have included the option of giving therapies at lower dose because we are interested in exploring how lower doses might be used, for the reasons illustrated in this game. There are some clinical trials now going on that are exploring the utility of giving lower doses (called "de-escalation trials"), but that is non-standard for most treatments. The initial version of this game included an immunotherapy (immune checkpoint blockade drugs), but we removed it because it isn't clear you can dial the immune system up and down. We think (roughly) that you either turn it on or you don't.

It is also the case that some treatments, like Herceptin, are not given in combination with any other treatments. While some treatments are routinely given in combinations (sometimes in combinations with more than 2 drugs!). 

#### Tests of biopsies (biomarker assays)
We have heavily simplified the tests (biomarker assays) in the game. Most biomarker tests not only show you what proportion of cells have the biomarker, like the estrogen receptor, but also how strongly those cells are expressing the biomarker (e.g., how much estrogen receptor is on a cell). We have completely left out this aspect of typical biomarker tests, and if you are a pathologist or an oncologist, this probably annoyed you and we apologize. In the game, cells are either positive or negative for the biomarker, and the test just returns a rough estimate of the proportion of cells that are positive.

You should also know that biopsies are typically only taken oncew, before therapy starts. In the game you can take a biopsy and run tests on it every month, and at no cost! If you want to be more realistic, then you should only run the tests at the start of the game, and then never run them again. Some oncologists are starting to order biopsies when the first treatment fails and they are trying to make a decision of what treatment to try next. Cancers change in response to the treatment, so you can't assume a new biopsy will be the same as the biopsy that was taken before treatment. But taking multiple biopsies during the course of treatment is not common yet, because we have not run the clinical trials to show that it would be useful. Tests that can be done on the blood (like the PSA test for prostate cancer) make this more feasible.

#### etc.
There are a host of other complexities we haven't represented, like the 3D structure of the tumor, drug delivery, half-life of the drugs, etc. An important one is that sometimes cancer cells go dormant. They stop dividing and go into a state that is extremely hard to kill, only to reawaken later for unknown reasons. We haven't represented that form of resistance. But the essential feature of cancers that make them hard to cure, that some cells are resistant to therapies, we have represented. And that has important therapeutic implications.

## EXTENDING THE MODEL

There is currently no explicit representation of the toxicity of therapies, except that no more than 2 therapies may be applied at any time. There is no cumulative toxicity either. For some treatments, the damage they do accumulates, which limits the total amount of drug that can be used on a patient over time. 

You could randomly allow some small proportion of cells to become dormant. They would neither divide nor participate in the cell competition, but they would be resistant to all therapies. Every month they would have a small probability of becoming proliferative again.

You might add complexities to address any of the other simplifications in the "THINGS THAT ARE UNREALISTIC" section.

## CREDITS, REFERENCES, LINKS AND CLINICAL TRIALS

This model was written by Prof. Carlo C. Maley. The game was developed in collaboration with Ellie Pahl, Bailey Kane, Jasmin Menjivar, Caroline Castle, Aya El Masry, and Dr. Lida Mina (a breast cancer oncologist). But please blame Carlo Maley for anything that you don't like. 

We have opened a clinical trial to test "adaptive therapy" in ER+ metastatic breast cancer that has become resistant to hormone therapy. At that point, patients would typically start a chemotherapy. We are testing if we can reduce the dose of the chemotherapy to keep sensitive cells around in order to compete with resistant cells, and thereby both keep patients alive longer and improve their quality of life (because lower dose means less toxicity). If you are interested in that trial, you can find out more about it at ***.

For breast cancers that are still sensitive to hormone therapy, there is also a clinical trial to test if using a lower dose hormone therapy works better than a high dose. You can find out about that trial at ***. 

If you would like to help fund our efforts to improve how we treat cancer, we would love it if you donated to our research efforts at ***.

And if you would like to volunteer to do cancer research with us, you can submit your interest and apply to our volunteer research program at https://acescholarsprogram.com

]]></info>
  <turtleShapes>
    <shape name="default" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="5"></point>
        <point x="40" y="250"></point>
        <point x="150" y="205"></point>
        <point x="260" y="250"></point>
      </polygon>
    </shape>
    <shape name="airplane" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="0"></point>
        <point x="135" y="15"></point>
        <point x="120" y="60"></point>
        <point x="120" y="105"></point>
        <point x="15" y="165"></point>
        <point x="15" y="195"></point>
        <point x="120" y="180"></point>
        <point x="135" y="240"></point>
        <point x="105" y="270"></point>
        <point x="120" y="285"></point>
        <point x="150" y="270"></point>
        <point x="180" y="285"></point>
        <point x="210" y="270"></point>
        <point x="165" y="240"></point>
        <point x="180" y="180"></point>
        <point x="285" y="195"></point>
        <point x="285" y="165"></point>
        <point x="180" y="105"></point>
        <point x="180" y="60"></point>
        <point x="165" y="15"></point>
      </polygon>
    </shape>
    <shape name="arrow" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="0"></point>
        <point x="0" y="150"></point>
        <point x="105" y="150"></point>
        <point x="105" y="293"></point>
        <point x="195" y="293"></point>
        <point x="195" y="150"></point>
        <point x="300" y="150"></point>
      </polygon>
    </shape>
    <shape name="box" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="285"></point>
        <point x="285" y="225"></point>
        <point x="285" y="75"></point>
        <point x="150" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="135"></point>
        <point x="15" y="75"></point>
        <point x="150" y="15"></point>
        <point x="285" y="75"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="15" y="75"></point>
        <point x="15" y="225"></point>
        <point x="150" y="285"></point>
        <point x="150" y="135"></point>
      </polygon>
      <line endX="150" startY="285" marked="false" color="255" endY="135" startX="150"></line>
      <line endX="15" startY="135" marked="false" color="255" endY="75" startX="150"></line>
      <line endX="285" startY="135" marked="false" color="255" endY="75" startX="150"></line>
    </shape>
    <shape name="bug" rotatable="true" editableColorIndex="0">
      <circle x="96" y="182" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="110" y="127" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <circle x="110" y="75" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <line endX="80" startY="100" marked="true" color="-1920102913" endY="30" startX="150"></line>
      <line endX="220" startY="100" marked="true" color="-1920102913" endY="30" startX="150"></line>
    </shape>
    <shape name="butterfly" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="165"></point>
        <point x="209" y="199"></point>
        <point x="225" y="225"></point>
        <point x="225" y="255"></point>
        <point x="195" y="270"></point>
        <point x="165" y="255"></point>
        <point x="150" y="240"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="165"></point>
        <point x="89" y="198"></point>
        <point x="75" y="225"></point>
        <point x="75" y="255"></point>
        <point x="105" y="270"></point>
        <point x="135" y="255"></point>
        <point x="150" y="240"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="139" y="148"></point>
        <point x="100" y="105"></point>
        <point x="55" y="90"></point>
        <point x="25" y="90"></point>
        <point x="10" y="105"></point>
        <point x="10" y="135"></point>
        <point x="25" y="180"></point>
        <point x="40" y="195"></point>
        <point x="85" y="194"></point>
        <point x="139" y="163"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="162" y="150"></point>
        <point x="200" y="105"></point>
        <point x="245" y="90"></point>
        <point x="275" y="90"></point>
        <point x="290" y="105"></point>
        <point x="290" y="135"></point>
        <point x="275" y="180"></point>
        <point x="260" y="195"></point>
        <point x="215" y="195"></point>
        <point x="162" y="165"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="255"></point>
        <point x="135" y="225"></point>
        <point x="120" y="150"></point>
        <point x="135" y="120"></point>
        <point x="150" y="105"></point>
        <point x="165" y="120"></point>
        <point x="180" y="150"></point>
        <point x="165" y="225"></point>
      </polygon>
      <circle x="135" y="90" marked="false" color="255" diameter="30" filled="true"></circle>
      <line endX="195" startY="105" marked="false" color="255" endY="60" startX="150"></line>
      <line endX="105" startY="105" marked="false" color="255" endY="60" startX="150"></line>
    </shape>
    <shape name="car" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="300" y="180"></point>
        <point x="279" y="164"></point>
        <point x="261" y="144"></point>
        <point x="240" y="135"></point>
        <point x="226" y="132"></point>
        <point x="213" y="106"></point>
        <point x="203" y="84"></point>
        <point x="185" y="63"></point>
        <point x="159" y="50"></point>
        <point x="135" y="50"></point>
        <point x="75" y="60"></point>
        <point x="0" y="150"></point>
        <point x="0" y="165"></point>
        <point x="0" y="225"></point>
        <point x="300" y="225"></point>
        <point x="300" y="180"></point>
      </polygon>
      <circle x="180" y="180" marked="false" color="255" diameter="90" filled="true"></circle>
      <circle x="30" y="180" marked="false" color="255" diameter="90" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="162" y="80"></point>
        <point x="132" y="78"></point>
        <point x="134" y="135"></point>
        <point x="209" y="135"></point>
        <point x="194" y="105"></point>
        <point x="189" y="96"></point>
        <point x="180" y="89"></point>
      </polygon>
      <circle x="47" y="195" marked="true" color="-1920102913" diameter="58" filled="true"></circle>
      <circle x="195" y="195" marked="true" color="-1920102913" diameter="58" filled="true"></circle>
    </shape>
    <shape name="circle" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
    </shape>
    <shape name="circle 2" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
    </shape>
    <shape name="cow" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="200" y="193"></point>
        <point x="197" y="249"></point>
        <point x="179" y="249"></point>
        <point x="177" y="196"></point>
        <point x="166" y="187"></point>
        <point x="140" y="189"></point>
        <point x="93" y="191"></point>
        <point x="78" y="179"></point>
        <point x="72" y="211"></point>
        <point x="49" y="209"></point>
        <point x="48" y="181"></point>
        <point x="37" y="149"></point>
        <point x="25" y="120"></point>
        <point x="25" y="89"></point>
        <point x="45" y="72"></point>
        <point x="103" y="84"></point>
        <point x="179" y="75"></point>
        <point x="198" y="76"></point>
        <point x="252" y="64"></point>
        <point x="272" y="81"></point>
        <point x="293" y="103"></point>
        <point x="285" y="121"></point>
        <point x="255" y="121"></point>
        <point x="242" y="118"></point>
        <point x="224" y="167"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="73" y="210"></point>
        <point x="86" y="251"></point>
        <point x="62" y="249"></point>
        <point x="48" y="208"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="25" y="114"></point>
        <point x="16" y="195"></point>
        <point x="9" y="204"></point>
        <point x="23" y="213"></point>
        <point x="25" y="200"></point>
        <point x="39" y="123"></point>
      </polygon>
    </shape>
    <shape name="cylinder" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
    </shape>
    <shape name="dot" rotatable="false" editableColorIndex="0">
      <circle x="90" y="90" marked="true" color="-1920102913" diameter="120" filled="true"></circle>
    </shape>
    <shape name="face happy" rotatable="false" editableColorIndex="0">
      <circle x="8" y="8" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="255"></point>
        <point x="90" y="239"></point>
        <point x="62" y="213"></point>
        <point x="47" y="191"></point>
        <point x="67" y="179"></point>
        <point x="90" y="203"></point>
        <point x="109" y="218"></point>
        <point x="150" y="225"></point>
        <point x="192" y="218"></point>
        <point x="210" y="203"></point>
        <point x="227" y="181"></point>
        <point x="251" y="194"></point>
        <point x="236" y="217"></point>
        <point x="212" y="240"></point>
      </polygon>
    </shape>
    <shape name="face neutral" rotatable="false" editableColorIndex="0">
      <circle x="8" y="7" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <rectangle endX="240" startY="195" marked="false" color="255" endY="225" startX="60" filled="true"></rectangle>
    </shape>
    <shape name="face sad" rotatable="false" editableColorIndex="0">
      <circle x="8" y="8" marked="true" color="-1920102913" diameter="285" filled="true"></circle>
      <circle x="60" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <circle x="180" y="75" marked="false" color="255" diameter="60" filled="true"></circle>
      <polygon color="255" filled="true" marked="false">
        <point x="150" y="168"></point>
        <point x="90" y="184"></point>
        <point x="62" y="210"></point>
        <point x="47" y="232"></point>
        <point x="67" y="244"></point>
        <point x="90" y="220"></point>
        <point x="109" y="205"></point>
        <point x="150" y="198"></point>
        <point x="192" y="205"></point>
        <point x="210" y="220"></point>
        <point x="227" y="242"></point>
        <point x="251" y="229"></point>
        <point x="236" y="206"></point>
        <point x="212" y="183"></point>
      </polygon>
    </shape>
    <shape name="fish" rotatable="false" editableColorIndex="0">
      <polygon color="-1" filled="true" marked="false">
        <point x="44" y="131"></point>
        <point x="21" y="87"></point>
        <point x="15" y="86"></point>
        <point x="0" y="120"></point>
        <point x="15" y="150"></point>
        <point x="0" y="180"></point>
        <point x="13" y="214"></point>
        <point x="20" y="212"></point>
        <point x="45" y="166"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="135" y="195"></point>
        <point x="119" y="235"></point>
        <point x="95" y="218"></point>
        <point x="76" y="210"></point>
        <point x="46" y="204"></point>
        <point x="60" y="165"></point>
      </polygon>
      <polygon color="-1" filled="true" marked="false">
        <point x="75" y="45"></point>
        <point x="83" y="77"></point>
        <point x="71" y="103"></point>
        <point x="86" y="114"></point>
        <point x="166" y="78"></point>
        <point x="135" y="60"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="30" y="136"></point>
        <point x="151" y="77"></point>
        <point x="226" y="81"></point>
        <point x="280" y="119"></point>
        <point x="292" y="146"></point>
        <point x="292" y="160"></point>
        <point x="287" y="170"></point>
        <point x="270" y="195"></point>
        <point x="195" y="210"></point>
        <point x="151" y="212"></point>
        <point x="30" y="166"></point>
      </polygon>
      <circle x="215" y="106" marked="false" color="255" diameter="30" filled="true"></circle>
    </shape>
    <shape name="flag" rotatable="false" editableColorIndex="0">
      <rectangle endX="75" startY="15" marked="true" color="-1920102913" endY="300" startX="60" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="90" y="150"></point>
        <point x="270" y="90"></point>
        <point x="90" y="30"></point>
      </polygon>
      <line endX="90" startY="135" marked="true" color="-1920102913" endY="135" startX="75"></line>
      <line endX="90" startY="45" marked="true" color="-1920102913" endY="45" startX="75"></line>
    </shape>
    <shape name="flower" rotatable="false" editableColorIndex="0">
      <polygon color="1504722175" filled="true" marked="false">
        <point x="135" y="120"></point>
        <point x="165" y="165"></point>
        <point x="180" y="210"></point>
        <point x="180" y="240"></point>
        <point x="150" y="300"></point>
        <point x="165" y="300"></point>
        <point x="195" y="240"></point>
        <point x="195" y="195"></point>
        <point x="165" y="135"></point>
      </polygon>
      <circle x="85" y="132" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="130" y="147" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="192" y="85" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="85" y="40" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="177" y="40" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="177" y="132" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="70" y="85" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="130" y="25" marked="true" color="-1920102913" diameter="38" filled="true"></circle>
      <circle x="96" y="51" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="113" y="68" marked="false" color="255" diameter="74" filled="true"></circle>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="189" y="233"></point>
        <point x="219" y="188"></point>
        <point x="249" y="173"></point>
        <point x="279" y="188"></point>
        <point x="234" y="218"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="180" y="255"></point>
        <point x="150" y="210"></point>
        <point x="105" y="210"></point>
        <point x="75" y="240"></point>
        <point x="135" y="240"></point>
      </polygon>
    </shape>
    <shape name="house" rotatable="false" editableColorIndex="0">
      <rectangle endX="255" startY="120" marked="true" color="-1920102913" endY="285" startX="45" filled="true"></rectangle>
      <rectangle endX="180" startY="210" marked="false" color="255" endY="285" startX="120" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="15" y="120"></point>
        <point x="150" y="15"></point>
        <point x="285" y="120"></point>
      </polygon>
      <line endX="270" startY="120" marked="false" color="255" endY="120" startX="30"></line>
    </shape>
    <shape name="leaf" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="210"></point>
        <point x="135" y="195"></point>
        <point x="120" y="210"></point>
        <point x="60" y="210"></point>
        <point x="30" y="195"></point>
        <point x="60" y="180"></point>
        <point x="60" y="165"></point>
        <point x="15" y="135"></point>
        <point x="30" y="120"></point>
        <point x="15" y="105"></point>
        <point x="40" y="104"></point>
        <point x="45" y="90"></point>
        <point x="60" y="90"></point>
        <point x="90" y="105"></point>
        <point x="105" y="120"></point>
        <point x="120" y="120"></point>
        <point x="105" y="60"></point>
        <point x="120" y="60"></point>
        <point x="135" y="30"></point>
        <point x="150" y="15"></point>
        <point x="165" y="30"></point>
        <point x="180" y="60"></point>
        <point x="195" y="60"></point>
        <point x="180" y="120"></point>
        <point x="195" y="120"></point>
        <point x="210" y="105"></point>
        <point x="240" y="90"></point>
        <point x="255" y="90"></point>
        <point x="263" y="104"></point>
        <point x="285" y="105"></point>
        <point x="270" y="120"></point>
        <point x="285" y="135"></point>
        <point x="240" y="165"></point>
        <point x="240" y="180"></point>
        <point x="270" y="195"></point>
        <point x="240" y="210"></point>
        <point x="180" y="210"></point>
        <point x="165" y="195"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="195"></point>
        <point x="135" y="240"></point>
        <point x="120" y="255"></point>
        <point x="105" y="255"></point>
        <point x="105" y="285"></point>
        <point x="135" y="285"></point>
        <point x="165" y="240"></point>
        <point x="165" y="195"></point>
      </polygon>
    </shape>
    <shape name="line" rotatable="true" editableColorIndex="0">
      <line endX="150" startY="0" marked="true" color="-1920102913" endY="300" startX="150"></line>
    </shape>
    <shape name="line half" rotatable="true" editableColorIndex="0">
      <line endX="150" startY="0" marked="true" color="-1920102913" endY="150" startX="150"></line>
    </shape>
    <shape name="pentagon" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="15"></point>
        <point x="15" y="120"></point>
        <point x="60" y="285"></point>
        <point x="240" y="285"></point>
        <point x="285" y="120"></point>
      </polygon>
    </shape>
    <shape name="person" rotatable="false" editableColorIndex="0">
      <circle x="110" y="5" marked="true" color="-1920102913" diameter="80" filled="true"></circle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="120" y="195"></point>
        <point x="90" y="285"></point>
        <point x="105" y="300"></point>
        <point x="135" y="300"></point>
        <point x="150" y="225"></point>
        <point x="165" y="300"></point>
        <point x="195" y="300"></point>
        <point x="210" y="285"></point>
        <point x="180" y="195"></point>
        <point x="195" y="90"></point>
      </polygon>
      <rectangle endX="172" startY="79" marked="true" color="-1920102913" endY="94" startX="127" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="195" y="90"></point>
        <point x="240" y="150"></point>
        <point x="225" y="180"></point>
        <point x="165" y="105"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="105" y="90"></point>
        <point x="60" y="150"></point>
        <point x="75" y="180"></point>
        <point x="135" y="105"></point>
      </polygon>
    </shape>
    <shape name="plant" rotatable="false" editableColorIndex="0">
      <rectangle endX="165" startY="90" marked="true" color="-1920102913" endY="300" startX="135" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="255"></point>
        <point x="90" y="210"></point>
        <point x="45" y="195"></point>
        <point x="75" y="255"></point>
        <point x="135" y="285"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="255"></point>
        <point x="210" y="210"></point>
        <point x="255" y="195"></point>
        <point x="225" y="255"></point>
        <point x="165" y="285"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="180"></point>
        <point x="90" y="135"></point>
        <point x="45" y="120"></point>
        <point x="75" y="180"></point>
        <point x="135" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="180"></point>
        <point x="165" y="210"></point>
        <point x="225" y="180"></point>
        <point x="255" y="120"></point>
        <point x="210" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="105"></point>
        <point x="90" y="60"></point>
        <point x="45" y="45"></point>
        <point x="75" y="105"></point>
        <point x="135" y="135"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="165" y="105"></point>
        <point x="165" y="135"></point>
        <point x="225" y="105"></point>
        <point x="255" y="45"></point>
        <point x="210" y="60"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="135" y="90"></point>
        <point x="120" y="45"></point>
        <point x="150" y="15"></point>
        <point x="180" y="45"></point>
        <point x="165" y="90"></point>
      </polygon>
    </shape>
    <shape name="sheep" rotatable="false" editableColorIndex="15">
      <circle x="203" y="65" marked="true" color="-1" diameter="88" filled="true"></circle>
      <circle x="70" y="65" marked="true" color="-1" diameter="162" filled="true"></circle>
      <circle x="150" y="105" marked="true" color="-1" diameter="120" filled="true"></circle>
      <polygon color="-1920102913" filled="true" marked="false">
        <point x="218" y="120"></point>
        <point x="240" y="165"></point>
        <point x="255" y="165"></point>
        <point x="278" y="120"></point>
      </polygon>
      <circle x="214" y="72" marked="false" color="-1920102913" diameter="67" filled="true"></circle>
      <rectangle endX="179" startY="223" marked="true" color="-1" endY="298" startX="164" filled="true"></rectangle>
      <polygon color="-1" filled="true" marked="true">
        <point x="45" y="285"></point>
        <point x="30" y="285"></point>
        <point x="30" y="240"></point>
        <point x="15" y="195"></point>
        <point x="45" y="210"></point>
      </polygon>
      <circle x="3" y="83" marked="true" color="-1" diameter="150" filled="true"></circle>
      <rectangle endX="80" startY="221" marked="true" color="-1" endY="296" startX="65" filled="true"></rectangle>
      <polygon color="-1" filled="true" marked="true">
        <point x="195" y="285"></point>
        <point x="210" y="285"></point>
        <point x="210" y="240"></point>
        <point x="240" y="210"></point>
        <point x="195" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="false">
        <point x="276" y="85"></point>
        <point x="285" y="105"></point>
        <point x="302" y="99"></point>
        <point x="294" y="83"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="false">
        <point x="219" y="85"></point>
        <point x="210" y="105"></point>
        <point x="193" y="99"></point>
        <point x="201" y="83"></point>
      </polygon>
    </shape>
    <shape name="square" rotatable="false" editableColorIndex="0">
      <rectangle endX="270" startY="30" marked="true" color="-1920102913" endY="270" startX="30" filled="true"></rectangle>
    </shape>
    <shape name="square 2" rotatable="false" editableColorIndex="0">
      <rectangle endX="270" startY="30" marked="true" color="-1920102913" endY="270" startX="30" filled="true"></rectangle>
      <rectangle endX="240" startY="60" marked="false" color="255" endY="240" startX="60" filled="true"></rectangle>
    </shape>
    <shape name="star" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="151" y="1"></point>
        <point x="185" y="108"></point>
        <point x="298" y="108"></point>
        <point x="207" y="175"></point>
        <point x="242" y="282"></point>
        <point x="151" y="216"></point>
        <point x="59" y="282"></point>
        <point x="94" y="175"></point>
        <point x="3" y="108"></point>
        <point x="116" y="108"></point>
      </polygon>
    </shape>
    <shape name="target" rotatable="false" editableColorIndex="0">
      <circle x="0" y="0" marked="true" color="-1920102913" diameter="300" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
      <circle x="60" y="60" marked="true" color="-1920102913" diameter="180" filled="true"></circle>
      <circle x="90" y="90" marked="false" color="255" diameter="120" filled="true"></circle>
      <circle x="120" y="120" marked="true" color="-1920102913" diameter="60" filled="true"></circle>
    </shape>
    <shape name="tree" rotatable="false" editableColorIndex="0">
      <circle x="118" y="3" marked="true" color="-1920102913" diameter="94" filled="true"></circle>
      <rectangle endX="180" startY="195" marked="false" color="-1653716737" endY="300" startX="120" filled="true"></rectangle>
      <circle x="65" y="21" marked="true" color="-1920102913" diameter="108" filled="true"></circle>
      <circle x="116" y="41" marked="true" color="-1920102913" diameter="127" filled="true"></circle>
      <circle x="45" y="90" marked="true" color="-1920102913" diameter="120" filled="true"></circle>
      <circle x="104" y="74" marked="true" color="-1920102913" diameter="152" filled="true"></circle>
    </shape>
    <shape name="triangle" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="30"></point>
        <point x="15" y="255"></point>
        <point x="285" y="255"></point>
      </polygon>
    </shape>
    <shape name="triangle 2" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="30"></point>
        <point x="15" y="255"></point>
        <point x="285" y="255"></point>
      </polygon>
      <polygon color="255" filled="true" marked="false">
        <point x="151" y="99"></point>
        <point x="225" y="223"></point>
        <point x="75" y="224"></point>
      </polygon>
    </shape>
    <shape name="truck" rotatable="false" editableColorIndex="0">
      <rectangle endX="195" startY="45" marked="true" color="-1920102913" endY="187" startX="4" filled="true"></rectangle>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="296" y="193"></point>
        <point x="296" y="150"></point>
        <point x="259" y="134"></point>
        <point x="244" y="104"></point>
        <point x="208" y="104"></point>
        <point x="207" y="194"></point>
      </polygon>
      <rectangle endX="195" startY="60" marked="false" color="-1" endY="105" startX="195" filled="true"></rectangle>
      <polygon color="255" filled="true" marked="false">
        <point x="238" y="112"></point>
        <point x="252" y="141"></point>
        <point x="219" y="141"></point>
        <point x="218" y="112"></point>
      </polygon>
      <circle x="234" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <rectangle endX="214" startY="185" marked="true" color="-1920102913" endY="194" startX="181" filled="true"></rectangle>
      <circle x="144" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <circle x="24" y="174" marked="false" color="255" diameter="42" filled="true"></circle>
      <circle x="24" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
      <circle x="144" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
      <circle x="234" y="174" marked="true" color="-1920102913" diameter="42" filled="false"></circle>
    </shape>
    <shape name="turtle" rotatable="true" editableColorIndex="0">
      <polygon color="1504722175" filled="true" marked="false">
        <point x="215" y="204"></point>
        <point x="240" y="233"></point>
        <point x="246" y="254"></point>
        <point x="228" y="266"></point>
        <point x="215" y="252"></point>
        <point x="193" y="210"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="195" y="90"></point>
        <point x="225" y="75"></point>
        <point x="245" y="75"></point>
        <point x="260" y="89"></point>
        <point x="269" y="108"></point>
        <point x="261" y="124"></point>
        <point x="240" y="105"></point>
        <point x="225" y="105"></point>
        <point x="210" y="105"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="105" y="90"></point>
        <point x="75" y="75"></point>
        <point x="55" y="75"></point>
        <point x="40" y="89"></point>
        <point x="31" y="108"></point>
        <point x="39" y="124"></point>
        <point x="60" y="105"></point>
        <point x="75" y="105"></point>
        <point x="90" y="105"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="132" y="85"></point>
        <point x="134" y="64"></point>
        <point x="107" y="51"></point>
        <point x="108" y="17"></point>
        <point x="150" y="2"></point>
        <point x="192" y="18"></point>
        <point x="192" y="52"></point>
        <point x="169" y="65"></point>
        <point x="172" y="87"></point>
      </polygon>
      <polygon color="1504722175" filled="true" marked="false">
        <point x="85" y="204"></point>
        <point x="60" y="233"></point>
        <point x="54" y="254"></point>
        <point x="72" y="266"></point>
        <point x="85" y="252"></point>
        <point x="107" y="210"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="119" y="75"></point>
        <point x="179" y="75"></point>
        <point x="209" y="101"></point>
        <point x="224" y="135"></point>
        <point x="220" y="225"></point>
        <point x="175" y="261"></point>
        <point x="128" y="261"></point>
        <point x="81" y="224"></point>
        <point x="74" y="135"></point>
        <point x="88" y="99"></point>
      </polygon>
    </shape>
    <shape name="wheel" rotatable="false" editableColorIndex="0">
      <circle x="3" y="3" marked="true" color="-1920102913" diameter="294" filled="true"></circle>
      <circle x="30" y="30" marked="false" color="255" diameter="240" filled="true"></circle>
      <line endX="150" startY="285" marked="true" color="-1920102913" endY="15" startX="150"></line>
      <line endX="285" startY="150" marked="true" color="-1920102913" endY="150" startX="15"></line>
      <circle x="120" y="120" marked="true" color="-1920102913" diameter="60" filled="true"></circle>
      <line endX="79" startY="40" marked="true" color="-1920102913" endY="269" startX="216"></line>
      <line endX="269" startY="84" marked="true" color="-1920102913" endY="221" startX="40"></line>
      <line endX="269" startY="216" marked="true" color="-1920102913" endY="79" startX="40"></line>
      <line endX="221" startY="40" marked="true" color="-1920102913" endY="269" startX="84"></line>
    </shape>
    <shape name="wolf" rotatable="false" editableColorIndex="0">
      <polygon color="255" filled="true" marked="false">
        <point x="253" y="133"></point>
        <point x="245" y="131"></point>
        <point x="245" y="133"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="2" y="194"></point>
        <point x="13" y="197"></point>
        <point x="30" y="191"></point>
        <point x="38" y="193"></point>
        <point x="38" y="205"></point>
        <point x="20" y="226"></point>
        <point x="20" y="257"></point>
        <point x="27" y="265"></point>
        <point x="38" y="266"></point>
        <point x="40" y="260"></point>
        <point x="31" y="253"></point>
        <point x="31" y="230"></point>
        <point x="60" y="206"></point>
        <point x="68" y="198"></point>
        <point x="75" y="209"></point>
        <point x="66" y="228"></point>
        <point x="65" y="243"></point>
        <point x="82" y="261"></point>
        <point x="84" y="268"></point>
        <point x="100" y="267"></point>
        <point x="103" y="261"></point>
        <point x="77" y="239"></point>
        <point x="79" y="231"></point>
        <point x="100" y="207"></point>
        <point x="98" y="196"></point>
        <point x="119" y="201"></point>
        <point x="143" y="202"></point>
        <point x="160" y="195"></point>
        <point x="166" y="210"></point>
        <point x="172" y="213"></point>
        <point x="173" y="238"></point>
        <point x="167" y="251"></point>
        <point x="160" y="248"></point>
        <point x="154" y="265"></point>
        <point x="169" y="264"></point>
        <point x="178" y="247"></point>
        <point x="186" y="240"></point>
        <point x="198" y="260"></point>
        <point x="200" y="271"></point>
        <point x="217" y="271"></point>
        <point x="219" y="262"></point>
        <point x="207" y="258"></point>
        <point x="195" y="230"></point>
        <point x="192" y="198"></point>
        <point x="210" y="184"></point>
        <point x="227" y="164"></point>
        <point x="242" y="144"></point>
        <point x="259" y="145"></point>
        <point x="284" y="151"></point>
        <point x="277" y="141"></point>
        <point x="293" y="140"></point>
        <point x="299" y="134"></point>
        <point x="297" y="127"></point>
        <point x="273" y="119"></point>
        <point x="270" y="105"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="-1" y="195"></point>
        <point x="14" y="180"></point>
        <point x="36" y="166"></point>
        <point x="40" y="153"></point>
        <point x="53" y="140"></point>
        <point x="82" y="131"></point>
        <point x="134" y="133"></point>
        <point x="159" y="126"></point>
        <point x="188" y="115"></point>
        <point x="227" y="108"></point>
        <point x="236" y="102"></point>
        <point x="238" y="98"></point>
        <point x="268" y="86"></point>
        <point x="269" y="92"></point>
        <point x="281" y="87"></point>
        <point x="269" y="103"></point>
        <point x="269" y="113"></point>
      </polygon>
    </shape>
    <shape name="x" rotatable="false" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="270" y="75"></point>
        <point x="225" y="30"></point>
        <point x="30" y="225"></point>
        <point x="75" y="270"></point>
      </polygon>
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="30" y="75"></point>
        <point x="75" y="30"></point>
        <point x="270" y="225"></point>
        <point x="225" y="270"></point>
      </polygon>
    </shape>
  </turtleShapes>
  <linkShapes>
    <shape name="default" curviness="0.0">
      <lines>
        <line x="-0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
        <line x="0.0" visible="true">
          <dash value="1.0"></dash>
          <dash value="0.0"></dash>
        </line>
        <line x="0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
      </lines>
      <indicator>
        <shape name="link direction" rotatable="true" editableColorIndex="0">
          <line endX="90" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
          <line endX="210" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
        </shape>
      </indicator>
    </shape>
  </linkShapes>
  <previewCommands>setup repeat 75 [ go ]</previewCommands>
</model>
